unit UnitACPopupPrototype;

interface

uses Classes, UnitClipQueue, System.Generics.Collections, Vcl.Controls,
Winapi.Messages, Vcl.ExtCtrls, Windows, UnitFrmTooltipNew, Vcl.Graphics,
UnitACPopupConfig, UnitMenuItemTagdata, UnitMisc,
System.Contnrs;

type
TACPopupItem = class;
TACPopupItemCollection = class;
TACPopupPrototype = class;
TACPopupItemClass = class of TACPopupItem;
TPopupItemStyle = (
    psNormal, psLine, psBreak, psSubmenu, psExpandable, psCheckable, psGap
);

TClipMenu = interface
        procedure setOnEditClip(ne : TNotifyEvent);
        procedure setOnRemoveClip(ne : TNotifyEvent);
        procedure setOnDestroyClip(ne : TNotifyEvent);
        procedure setOnPasteClip(ne : TNotifyEvent);
        procedure setOnMakePermanentClip(ne : TNotifyEvent);
        procedure setOnFormMode(ne : TNotifyEvent);
        procedure setOnHide(ne : TNotifyEvent);


        procedure SetACPopupItem(p : TACPopupItem);
        function GetACPopupItem : TACPopupItem;
        procedure SetPermanentClip(group, index : integer; format : TClipFormatType);
        procedure SetPopupClipMode;
        procedure SetPermanentClipMode;
        procedure Show;
        procedure Hide;

        procedure ReportKeypress(key : Char); overload;
        procedure ReportKeypress(key : Word); overload;
        procedure SetCustomButton(index: integer; caption, clip : string);

        function MouseInMenu : boolean;

        procedure PerformDefaultAction;
end;
TPopupItemPosition = (
    pipLeftIcon, pipRightIcon, pipCaption, pipNone
);

TACIcon = class(TObject)
private
    fIcon : HICON;
    fIconBitmap : TBitmap;
    fIconOpacity : byte;
    fBitmap : TBitmap;
    fCliptypeBitmap : TBitmap;
    fHoverBitmap : TBitmap;
    fHasCliptypeIcon : boolean;
    fHasIcon : boolean;

    fIconOnlyOnHover : boolean;
    fIconOnExpanded : boolean;

    procedure SetIcon(value : hicon);
    procedure SetBitmap(value : TBitmap);
    procedure SetCliptypeIcon(value : TBitmap);
    procedure SetHoverBitmap(value : TBitmap);
public
    constructor Create();
    destructor Destroy(); override;
    procedure SetFrom(icon : HICON); overload;
    procedure SetFrom(Bitmap : TBitmap); overload;

    property Icon : HICON read fIcon write SetIcon;
    property Bitmap : TBitmap read fBitmap write SetBitmap;
    property CliptypeIcon : TBitmap read fCliptypeBitmap write SetCliptypeIcon;
    property HoverBitmap : TBitmap read fHoverBitmap write SetHoverBitmap;
    property IconOnlyOnHover : boolean read fIconOnlyOnHover write fIconOnlyOnHover;
    property IconOpacity : Byte read fIconOpacity write fIconOpacity;

end;
TACPopupItemCanExpand = function (Sender: TACPopupItem) : boolean of object;
TACPopupItemCanCollapse = TACPopupItemCanExpand;

TACPopupItem = class(TObject)
private
    fLeftIcon : TACIcon;
    fCaptionTrailingBitmap : TBitmap;
    fCaptionIconOnlyOnHover : boolean;
    fCaption : string;
    fHoverOnlyCaption : string;
    fIconHoverCaption : string;
    fPrefix : string;

    fVisible : boolean;
    fSeparatorLine: boolean;
    fBottomLine: boolean;
    fSmallCaption: boolean;
    fStayOpenOnClick : boolean;

    fClip : TClipItem;
    fItemType : TItemType;
    fPermanentID, fPermanentGroupID : integer;
    fPermanentType : TClipFormatType;

    fMaxHeight : integer;
    fColWidth : TIntList;
//    fColWidth : TList<integer>;
    fSubMenu : TACPopupItemCollection;
    fCheckgroup : TObjectList;
    fParent : TACPopupItem;
    fMyColumn : integer;

    fSubShowing : TACPopupItem;
    fSubRect : TRect;

    fExpanded : boolean;
    fCollapseParentOnClick : boolean;
    fChecked : boolean;

    fStyle : TPopupItemStyle;
    fBoundsRect : TRect;
    fDoubleHeight : boolean;
    fDisabled : boolean;
    fShowCollapseIcon : Boolean;

    fCanExpand : TACPopupItemCanExpand;
    fCanCollapse : TACPopupItemCanCollapse;
    fStaticRight : boolean;
    fNoSmallHeight : boolean;
    fClickableLeftIcon : boolean;
    fac : TACPopupPrototype;
    function IsOnClipboard : boolean;
    function IsPictureClip : boolean;
    function getClickableLeftIcon : boolean;
protected
    fBottom : integer;
    fLeft : integer;
    fTop : integer;
    fRight : integer;
    fOnClick : TNotifyEvent;
    fOnRightClick : TNotifyEvent;
    fctrlpressed, fShiftPressed, fMiddleClicked, fTabPressed, fAltpressed : boolean;
    fIntegerData : integer;

    fScaleLeftBitmapToHeight : boolean;

    fHint : string;
    fCaptionBold : Boolean;
    fUseAutoPrefix : boolean;
    procedure SetCaption(value : string);

    function CalcRight : integer;
    function GetIsInSubMenu : boolean;
    function GetIsInExpandedMenu : boolean;
    //function GetColumnWidth : integer;
    function GetChecked : boolean;
    procedure SetChecked(value : boolean);

    function GetIsLine : boolean;
    procedure SetIsLine(value : boolean);
    function GetIsBreak : boolean;
    procedure SetIsBreak(value : boolean);
    function GetIsExpandable : boolean;
    procedure SetIsExpandable(value : boolean);
    function GetIsSubmenu : boolean;
    procedure SetIsSubmenu(value : boolean);
    function GetIsGap : boolean;
    procedure SetIsGap(value : boolean);

    function GetItem(Index : integer) : TACPopupItem;
    procedure SetItem(Index : integer; value : TACPopupItem);
    function GetExpanded : boolean;
    procedure SetExpanded(value : boolean);
    function GetIndex : integer;
    function GetOwnerIndex : integer;
    function GetVisibleParent : TACPopupItem;

    procedure SetBoundsRect(parenttop, parentleft : integer);
    function GetBoundsRect : TRect;
    procedure SetCaptionFromClip(value : TClipItem);


    procedure SetHoverOnlyCaption(value : string);
    function GetHoverOnlyCaption : string;

    procedure getTextAreas(c : TCanvas; mainr : TRect; var prefix : TRect; var caption : TRect);
    function getPrefixWidth(C : TCanvas) : Integer;

    function getVisible : boolean;
    function showPrefix(ShowCaptionShortcuts : Boolean ): boolean;


public

    constructor Create();
    destructor Destroy; override;

    property LeftIcon : TACIcon read fLeftIcon;
    function Add : TACPopupItem;
    function AddBreak : TACPopupITem;
    procedure CheckGrouped;
    function IsCheckGrouped : boolean;
    procedure SetShowingSubMenu;
    function DesiredWidth(ac : TACPopupPrototype; c : TCanvas) : integer;
    procedure DrawCaption(AC : TACPopupPrototype; c : TCanvas; r : TRect; IsHovered, IsIcon :  boolean);
    procedure DrawCaptionBackground(AC: TACPopupPrototype; c : TCanvas; r : TRect);

    property Right : integer read CalcRight;
    property Left : integer read fLeft;
    property Top : integer read fTOp;
    property Bottom : integer read fBottom;
    property BoundRect : TRect read GetBoundsRect;
    property Clip : TClipItem read fClip write fClip;

    property Caption : String read fCaption write SetCaption;
    property CaptionRaw : string write fCaption;
    property CaptionFromClip : TClipItem write SetCaptionFromClip;
    property CaptionBold : Boolean read fCaptionBold write fCaptionBold;


    property IsLine : boolean read GetIsLine write SetIsLine;
    property IsBreak : boolean read GetIsBreak write SetIsBreak;
    property IsExpandable : boolean read GetIsExpandable write SetIsExpandable;
    property IsSubmenu : boolean read GetIsSubmenu write SetIsSubmenu;
    property isGap : boolean read GetIsGap write SetIsGap;
    property Checked : boolean read GetChecked write SetChecked;
    property Expanded : boolean read GetExpanded write SetExpanded;
    property Style : TPopupItemStyle read fStyle;

    property OnClick : TNotifyEvent read fOnClick write fOnClick;
    property OnRightClick : TNotifyEvent read fOnRightClick write fOnRightClick;

    property Prefix : string read fPrefix write fPrefix;
    property ItemType : TItemType read fItemType write fItemType;
    property Visible : boolean read getVisible write fVisible;
    property IsInSubmenu : boolean read GetIsInSubMenu;
    property IsInExpandedMenu : boolean read GetIsInExpandedMenu;
    //property ColumnWidth : integer read GetColumnWidth;
    property ColumnIndex : integer read fMyColumn;
    property Items[Index : integer] : TACPopupItem read GetItem write Setitem; default;
    property Index : integer read GetIndex;
    property OwnerIndex : integer read GetOwnerIndex;
    property VisibleParent : TACPopupItem read GetVisibleParent;
    property CtrlPressed : boolean read fCtrlPressed write fCtrlPressed;
    property ShiftPressed : boolean read fShiftPressed write fShiftPressed;
    property MiddleClicked : Boolean read fMiddleClicked write fMiddleClicked;
    property AltPressed : boolean read fAltpressed write fAltpressed;
    property TabPressed : Boolean read fTabPressed write fTabPressed;

    property IntegerData : integer read fIntegerData write fIntegerData;
    property PermanentGroupID : integer read fPermanentGroupID write fPermanentGroupID;
    property PermanentID : integer read fPermanentID write fPermanentID;
    property PermanentType : TClipFormatType read fPermanentType write fPermanentType;
    property Hint : string read fHint write fHint;
    property DoubleHeight : boolean read fDoubleHeight write fDoubleHeight;
    property Disabled : boolean read fDisabled write fDisabled;

    property ShowCollapseIcon : Boolean read fShowCollapseIcon write fShowCollapseIcon;
    property CanExpand : TACPopupItemCanExpand read fCanExpand write fCanExpand;
    property CanCollapse : TACPopupItemCanExpand read fCanCollapse write fCanCollapse;
    property StayOpenOnClick : boolean read fStayOpenOnClick write fStayOpenOnClick;
    property SeparatorLine : Boolean read fSeparatorLine write fSeparatorLine;
    property BottomLine : Boolean read fBottomLine write fBottomLine;
    property SmallCaption : Boolean read fSmallCaption write fSmallCaption;
    property NoSmallHeight : boolean read fNoSmallHeight write fNoSmallHeight;

    property HoverOnlyCpation : string read getHoverOnlyCaption write setHoverOnlyCaption;

    property UseAutoPrefix : Boolean read fUseAutoPrefix write fUseAutoPrefix;
    property ClickableLeftIcon : Boolean read getClickableLeftIcon write fClickableLeftIcon;
    property SubMenu : TACPopupItemCollection read fSubMenu;
    property CaptionTrailingBitmap : TBitmap read fCaptionTrailingBitmap write fCaptionTrailingBitmap;
    property IconHoverCaption : string read fIconHoverCaption write fIconHoverCaption;
    property CollapseParentOnClick : Boolean read fCollapseParentOnClick write fCollapseParentOnClick;
    property CaptionIconOnlyOnHover : Boolean read fCaptionIconOnlyOnHover write fCaptionIconOnlyOnHover;
    property SubShowing : TACPopupItem read fSubShowing write fSubShowing;
    property Parent : TACPopupItem read fParent;
end;
TACPopupItemCollection = class(TObjectList)
protected
    function GetItem(Index: Integer): TACPopupItem;
    procedure SetItem(Index: Integer; const Value: TACPopupItem);
public
    function Add : TACPopupItem; overload;
    property Items[Index: Integer]: TACPopupItem read GetItem write SetItem; default;
end;

TPopupDisplayMode = (
    pdmAutoHide, pdmFormMode, pdmStayOpen, pdmPinned
);
TACPopupMode = class(TObject)
    private
        fbaseMode : TPopupDisplayMode;
        flist : TStack<TPopupDisplayMode>;


    public
        constructor Create;
        destructor Destroy; override;

        procedure setMode(mode : TPopupDisplayMode);
        function getMode : TPopupDisplayMode;
        function IsAutoClose : boolean;
        function equals(mode : TPopupDisplayMode) : boolean;
        procedure SetOverride(mode : TPopupDisplayMode);
        procedure ClearOverride;
        function getModeNoOverride : TPopupDisplayMode;
        procedure reset;

        procedure PushValue;
        function PopValue : TPopupDisplayMode;
end;
TCustomTransparentControl = class(TCustomControl)
  private
    FInterceptMouse: Boolean;
  protected
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure CreateParams(var Params: TCreateParams); override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Invalidate; override;
    property InterceptMouse: Boolean read FInterceptMouse write FInterceptMouse default False;
  end;
TACPopupPrototype = class(TCustomTransparentControl)
    imgPasteMini: TImage;
    procedure CreateParams(var Params: TCreateParams);  override;
private
    timMouseSelectDelay: TTimer;

	crit : RTL_CRITICAL_SECTION;
    scroll : THandle;
    timHover: TTimer;
    timShowSubmenu: TTimer;
    timHideSubmenu: TTimer;
    timLeftIconHover : TTimer;
    timIgnoreMouseMove : TTimer;
    timModifier : TTimer;
    ignoreHitTestOnce : boolean;

    fFirstPaint : boolean;

    LastMouseX, LastMouseY : integer;
    fItems : TACPopupItem;
    fTools : TACPopupItem;
    fLastVisible : TACPopupItem;

    fKeystrokes : TStringList;
    fAltKeyDown : boolean;
    fMaxItemWidth : integer;

    fFullMode : boolean;
    fHover : TACPopupItem;
    fDownShift : TShiftState;
    FLastPosition : TPopupItemPosition;
    fToolTipNew : TFrmTooltipNew;

    fAcceleratorCount : integer;
    fAccelCharsUsed : string;
    fReservedCount : integer;
    fLastSubmenu : TACPopupItem;
    fQueuedSubmenu : TACPopupItem;
    fMainRect : TRect;


    fPoint : TPoint;
    fTopFixed, fLeftFixed : boolean;

    region1, region2 : HRGN;
    fMouseDownOn : TACPopupItem;

    fNewCanvas : TCanvas;
    fDragStarted : boolean;
    fDragMenuItem : TACPopupItem;
    fDropMenuItem : TACPopupItem;
    fDragX, fDragY : integer;

    fOnHideEvent : TNotifyEvent;

    fIncludeEmpty : boolean;

    fIsTooTall : boolean;
    fTopOffset : integer;
    fTopOffsetEnable : boolean;
    fIgnoreMouseMove : boolean;
    fIgnoreKeypresses : boolean;
    fLastKeyDown : word;
    fLastModiferDown : TShiftState;




    //

    fLastClipSequence : cardinal;
    //fTempBitmap : TBitmap;

    fCaptionPoint : TPoint;
    fKeyActivated : boolean;
    fShowExpandedShortcuts : Boolean;
    fExpandedKeystrokes : TStringList;


    fAddLineToNext : boolean;
    fForceHide : boolean;

    flastRebuildUsedKeyboard : boolean;

    fTotalShortcutKeysAvailable : integer;
    fCaptionHeight : integer;

    fShowCaptionShortcuts : boolean;

    fleftToolsList : TList<TACPopupItem>;

    toolsMinWidth : integer;
    lastActiveState : integer;
    fConfig : TACConfig;


    procedure timMouseSelectDelayTimer(Sender: TObject);
    procedure timHoverTimer(Sender: TObject);
    procedure timShowSubmenuTimer(Sender: TObject);
    procedure timHideSubmenuTimer(Sender: TObject);
    procedure timLeftIconHoverTimer(Sender: TObject);
    procedure timIgnoreMouseMoveTimer(Sender: TObject);
    procedure timModiferTimer(Sender: TObject);


    procedure StartHoverTooltip;

    procedure WMGetDlgCode(var Msg: TMessage); message WM_GETDLGCODE;
    procedure WMActivate(Var msg:tMessage); message WM_ACTIVATE;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(Var M: TWMSetCursor); message WM_SETCURSOR;
	procedure WMMove(var Msg: TMessage) ; message WM_MOVE;


    function CalcToolWidth(pi : TACPopupItem) : integer;

    function CalcHeight(fpi : TACPopupItem) : integer;

    procedure MouseLeave(Sender: TObject);



    procedure ReserveAccel;
    procedure DetectAutoScroll(top,bottom : integer);


    procedure SetHoverUnderMouse;
    procedure SetFormMode(value : boolean);
    function GetFormMode : boolean;

    function GetSpacing : integer;
    function getBottomBorder : integer;
    function getLeftEdge : integer;
protected
    fUseExpandedShortcuts : boolean;
    fPopupMode : TACPopupMode;
    fShowing : boolean;
    fShowShadow : boolean;
    fCanFocus : boolean;
    ShowCaption : boolean;
    fBackgroundColor,
    fHighlightColor,
    fFontColor,
    fClickedColor,
    fDisabledColor,
    fExpandedBackgroundColor,
    fColumnColor,
    fBorderColor,
    fColumnEdgeColor,
    fBackgroundBase,
    fCaptionColor : TColor;

    procedure setConfig(config : TACConfig);

    procedure paint; override;
    procedure resize; override;

    function UseKeyboard : boolean; virtual;
    function UsePrefix : boolean; virtual;

    procedure GatherModifierKeys(p : TACPopupItem);
    procedure DetectKeystroke(p : TACPopupItem; useUppercase : Boolean = true); virtual;
    procedure DrawItem(p : TACPopupItem);
    procedure DrawCheckgroup(p : TACPopupItem);
    procedure FullRedraw;

    procedure DrawDragmark;
    procedure MyFillRect(r : TRect);
    procedure ClipMenuShow(rightclick : boolean = false); virtual; abstract;
    procedure ClipMenuHide; virtual; abstract;
    procedure ClipMenuReportKey(key : Char); overload; virtual; abstract;
    procedure ClipMenuReportKey(key : word); overload; virtual; abstract;
    function ClipMenuVisible : boolean; virtual; abstract;
    procedure HideTooltip; virtual;
    procedure RebuildPopup(DoAutoPopulate : boolean=true);
    function CalcCaptionHeight : integer;
    procedure HoverUp;
    procedure HoverLeft;
    procedure HoverRight;
    function isTool(p : TACPopupItem) : boolean;
    procedure HandleOnClick(p : TACPopupItem); virtual;
    procedure PreHandleCaptionViaKeystroke(p : TACPopupITem; key : char); virtual; abstract;
    function HandleMouseClick(p : TACPopupItem; DownShift : TShiftState):boolean; virtual;
    procedure HandleCaptionClick(p : TACPopupItem); virtual;
    procedure HoverClick;
    procedure ShowSubmenuDelayed(p : TACPopupItem);
    procedure CancelShowSubmenu;
    procedure MouseMove(Shift: TShiftState; X: Integer; Y: Integer); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X: Integer; Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X: Integer; Y: Integer); override;

    function GetItemAt(x,y : integer) : TACPopupItem;
    function MouseInPopup : boolean;
    procedure SetHover(value : TACPopupItem; tooltip:boolean=true; redraw:Boolean=true);  virtual;
    procedure ClearHover;
    property Hover : TACPopupItem read fHover;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;

    procedure EditCallback(Sender : Tobject);
    procedure DeleteClipCallback(Sender : TObject);
    procedure DestroyCallback(Sender : Tobject);
    procedure PasteCallback(Sender : Tobject);
    procedure MakePermanentCallback(Sender : Tobject);
    procedure FormModeCallback(Sender : TObject);
    procedure HideCallback(Sender : TObject);


    function IsLegalDrop(p, source : TACPopupItem) : boolean; virtual;
    function IsLegalDrag(p : TACPopupItem) : boolean; virtual;
    procedure DragMouseDown(p : TACPopupItem);
    function DragMouseMove(p, last : TACPopupItem) : boolean;
    function DragMouseUp(p : TACPopupItem) : boolean;
    procedure DragHandleDrop(source, target : TACPopupItem); virtual;

    function GetCanvas : TCanvas;
    procedure SetShowExpandedShortcuts(value : Boolean);
    property ShowExpandedShortcuts : boolean read fShowExpandedShortcuts write SetShowExpandedShortcuts;
    property ForceHide : boolean read fForceHide write fForceHide;
    property LastPosition : TPopupItemPosition read FLastPosition write FLastPosition;
    //procedure CheckForAutoExpand(p : TACPopupItem);  virtual; abstract;
    procedure ShowTooltip(p : TACPopupItem); virtual;
    procedure ActiveEvent(clicked : Boolean); virtual;
    procedure InactiveEvent; virtual;
    function CalcWidth(fpi : TACPopupItem; ClipIt : boolean=true; MinWidth : boolean=true) : integer; virtual;
    function CalcIconRect(p : TACPopupItem; bounds : TRect) : TRect; virtual;
    function CalcIconBackgroundRect(p : TACPopupItem; bounds : TRect) : TRect; virtual;
    procedure ClearMainKeystrokes;

    function IsDoubleHeightType(p : TACPopupItem) : boolean; virtual;
    procedure AssignSubmenuShortcuts(p : TACPopupItem); virtual;

    function getLeftSpaceWidth(p : TACPopupItem) : integer; virtual;

    function GetAccelerator(i: integer; NoReserve : boolean = false): char;
    function GetAcceleratorReserved(i: integer) : Char; virtual;

    procedure PostSizeCalculationEvent; virtual;
    procedure PostDrawIconEvent(p : TACPopupItem; r : TRect); virtual;
    function getDrawLeftIcon(p : TACPopupItem) : Boolean; virtual;

    procedure MouseCursorReset;
    procedure CancelHoverTimer;
    procedure CancelModifierTimer;
    procedure CancelLeftIconTimer;
    property SPACING : Integer read getSpacing;
    //property BOTTOMBORDER : Integer read getBottomBorder;
    property LEFTEDGE : integer read getLeftEdge;
    property ShowCaptionShortcuts : Boolean read fShowCaptionShortcuts write fShowCaptionShortcuts;
    property LastKeyDown : word read fLastKeyDown write fLastKeyDown;

    property Items : TACPopupItem read fItems;

    property LastRebuildUsedKeyboard : Boolean read flastRebuildUsedKeyboard;
    property Keystrokes : TStringList read fKeystrokes;
    property TotalShortcutKeysAvailable : integer read fTotalShortcutKeysAvailable write fTotalShortcutKeysAvailable;
    property AccelCharsUsed : string read fAccelCharsUsed write fAccelCharsUsed;
    property AcceleratorCount : Integer read fAcceleratorCount write fAcceleratorCount;
    property Tools : TACPopupItem read fTools;
    property LeftToolsList : TList<TACPopupItem> read fLeftToolsList;
    property Config : TACConfig read fConfig write setConfig;
    property CaptionHeight : Integer read fCaptionHeight write fCaptionHeight;
    property AddLineToNext : boolean read fAddLineToNext write fAddLineToNext;
    property CaptionPoint : TPoint read fCaptionPoint write fCaptionPoint;
    property KeyActivated : boolean read fKeyActivated;
    property ReservedCount : integer read fReservedCount;
    property ToolTipNew : TFrmTooltipNew read fToolTipNew;
    property FirstPaint : Boolean read fFirstPaint;
public

    { Creation }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy;  override;

    function Add : TACPopupItem;
    procedure AddLine;
    function AddBreak : TACPopupItem;

    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;

    procedure AutoPopulate; virtual; abstract;
    procedure CalcTotalSize;
    procedure Reset; virtual;
    procedure ShowPopup(x,y : integer; NoSizeCalc : boolean = false); overload; virtual;
    procedure ShowPopup(pt : TPoint); overload;
    procedure DodgePoint(pt : TPoint);
    procedure HoverDown;
    procedure Hide; virtual;

    {External Draw Routines}
    procedure SetCanvas(c : TCanvas);
    procedure DrawOnCanvas(fullmode : boolean=false);
    //property UseCanvas : TCanvas read GetCanvas;
    property IncludeEmpty : boolean read fIncludeEmpty write fIncludeEmpty;

    { Cosmetics }
    //property Spacing : integer read fSPacing write fSpacing;
    property MaxItemWidth : integer read fMaxItemWidth write fMaxItemWidth;
    property OnHideEvent : TNotifyEvent read fOnHideEvent write fOnHideEvent;
    property Showing : boolean read fShowing;
    property FullMode : boolean read fFullMode write fFullMode;
    //property UseGrouping : boolean read fUseGrouping write fUseGrouping;
    //property ClipsPerSet : integer read fGroupingSize write fGroupingSize;

    property UseFormMode : boolean read GetFormMode write SetFormMode;
end;


implementation

uses SysUtils, VCL.forms, UnitFrmDebug, UnitFocusManager, Math, UnitTWideChar,
GraphUtil,UnitKeyboardQuery, UnitToken, UnitFrmMainPopup, UnitPopupGenerate,
  UnitFrmConfig, ShellAPI, StrUtils;

{$R *.dfm}

const BLANK_SPACING = 2;
const ICON_WIDTH = 16;
const ICON_HEIGHT = ICON_WIDTH;
const LEFT_SPACE = BLANK_SPACING+ICON_WIDTH + BLANK_SPACING+BLANK_SPACING;
const POPUP_LEFT_BORDER = 1;
const MENU_LEFT_EDGE = 1;
const PREFIX_TRAILING_SPACE_MULTIPLE = 2;
const RIGHT_SPACE = 20;
const RIGHT_SPACE_PADDING_MULTIPLE = 3;
const POPUP_RIGHT_BORDER = 1;
const POPUP_TOP_BORDER = 1;
const POPUP_BOTTOM_BORDER = 1;
const MENU_CAPTION_LEFT = LEFT_SPACE+MENU_LEFT_EDGE;
const HEIGHT_PADDING = 1;
const LINE_HEIGHT = 6;
const RIGHT_MENU_GRADIENT_WIDTH = 35;

function TACPopupPrototype.GetSpacing : integer;
begin
    result := BLANK_SPACING;
end;
function TACPopupPrototype.getBottomBorder;
begin
    result := POPUP_BOTTOM_BORDER;
end;
function TACPopupPrototype.getLeftEdge;
begin
    result := MENU_LEFT_EDGE;
end;


var itemCnt : integer;
procedure TACPopupPrototype.CreateParams(var Params: TCreateParams);
begin
    inherited CreateParams(params);

   	Params.Style := WS_POPUP or (WS_CLIPCHILDREN or WS_CLIPSIBLINGS) ;

    Params.ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST;


    // setting the WS_EX_APPWINDOW at first lets the window toggle back to
    // this mode and immediately show on the taskbar

    // create() will change the mode to WS_EX_TOOLWINDOW


    {WS_EX_COMPOSITED WS_EX_TRANSPARENT screws up in WinXP - causing a blank redraw
    WS_EXPOMPOSTIED cannot be used in Aero}
end;
constructor TACPopupPrototype.Create(AOwner: TComponent);
begin


    fConfig := TACDefaultConfig.Create();
    self.InterceptMouse := true;
    self.Visible := false;
    fReservedCount := 0;
    fCanFocus := true;

    // NOTE, inherited was moved here for a reason

    inherited;
    if Win32MajorVersion >= 6 then begin
    	self.DoubleBuffered := true; // CANT use this with transparent on XP
    end;

    fPopupMode := TACPopupMode.Create;

    fBackgroundColor := ColorToRGB(clBtnFace);
    self.fColumnColor := fBackgroundColor;
    fHighlightColor := ColorToRGB(clHighlight);
    fFontColor := ColorToRGB(clBtnText);
    fExpandedBackgroundColor := BlendOrReverse(fBackgroundColor, clWhite, clBlack, 90);
    fColumnColor := BlendOrReverse(fBackgroundColor, $000000, clWhite, 93);
    fColumnEdgeColor := DimColorOrReverse(fBackgroundColor,0.86);
    fDisabledColor := ColorToRGB(clGrayText);

    fCaptionColor := fColumnColor;


    self.OnExit := self.MouseLeave;

    self.Width := 1;
    self.Height := 1;

    fToolTipNew := TFrmTooltipNew.Create(application);
    timhover := TTimer.Create(self);
    CancelHovertimer;

    timHover.OnTimer := self.timHoverTimer;
    timHideSubmenu := TTimer.Create(self);
    timHideSubmenu.Enabled := false;
    timHideSubmenu.Interval := 300;
    timHideSubmenu.OnTimer := self.timHideSubmenuTimer;
    timShowSubmenu := TTimer.Create(self);
    timShowSubmenu.Enabled := false;
    timShowSubmenu.Interval := 300;
    timShowSubmenu.OnTimer := self.timShowSubmenuTimer;

    timLeftIconHover := TTimer.Create(self);
    timLeftIconHover.Enabled := false;
    timLeftIconHover.Interval := 190;
    timLeftIconHover.ontimer :=  timLeftIconHoverTimer;

    timIgnoreMouseMove := TTimer.Create(self);
    timIgnoreMouseMove.Enabled := false;
    timIgnoreMouseMove.Interval := 100;
    timIgnoreMouseMove.OnTimer := timIgnoreMouseMoveTimer;

    timMouseSelectDelay := TTimer.Create(self);
    timMouseSelectDelay.Enabled := false;
    timMouseSelectDelay.Interval := 300;
    timMouseSelectDelay.OnTimer := timMouseSelectDelayTimer;

    timModifier := TTimer.Create(self);
    CancelModifiertimer;
    timModifier.Interval := 300;
    timModifier.OnTimer := timModiferTimer;


    fItems := TACPopupItem.Create();
    fTools := TACPopupItem.Create();
    fKeystrokes := TStringList.create;
    fExpandedKeystrokes := TStringList.Create;
    fExpandedKeystrokes.OwnsObjects := false;
    fExpandedKeystrokes.Duplicates := dupError;
    fUseExpandedShortcuts := True;

    self.fMaxItemWidth := 300;

    self.Width := 1;
    self.Height := 1;

    self.ParentColor := false;         /// MUST BE FALSE for transparent
    self.ParentBackground := false;    /// ditoo++

    timHover.Interval := Application.HintPause;
    fBorderColor := saturate(cl3DDkShadow,0.10);


    fleftToolsList := TList<TACPopupItem>.create;

    Windows.InitializeCriticalSection(crit);
    scroll := 0;
end;
destructor TACPopupPrototype.Destroy;
var i : integer;
    ac : TACPopupItem;
begin
    FrmDebug.AppendLog('UnitACPopupPrototype Destroy', false);

    self.Reset;

    if assigned(fTooltipNew) then
        fToolTipNew.close();
    Windows.DeleteCriticalSection(crit);

    FreeAndNil(fPopupMode);
    FreeAndNil(fTools);
    FreeAndNil(fItems);
    FreeAndNil(fKeystrokes);
    FreeAndNil(fExpandedKeystrokes);
    FreeAndNil(fleftToolsList);

    fLastVisible := nil;
    fDropMenuItem := nil;
    fDragMenuItem := nil;
    fQueuedSubmenu := nil;
    fLastSubmenu := nil;
    fHover := nil;

    inherited;

    FrmDebug.AppendLog('acpopupitem cnt = ' + IntToStr(itemCnt));
end;
procedure TACPopupPrototype.setConfig(config : TACConfig);
begin
    //if Assigned(fConfig) then FreeAndNil(fConfig);
    fConfig := config;
end;

procedure TACPopupPrototype.CancelLeftIconTimer;
begin
    timLeftIconHover.Enabled := false;
end;
procedure TACPopupPrototype.CancelModifierTimer;
begin
    timModifier.Enabled := false;
end;
procedure TACPopupPrototype.CancelHoverTimer;
begin
    timHover.Enabled := false;
end;



function TACPopupPrototype.Add: TACPopupItem;
begin
    result := fItems.add;
    if AddLineToNext then begin
        AddLineToNext := false;
        result.SeparatorLine := true;
    end;
end;



procedure TACPopupPrototype.ShowPopup(pt : TPoint);
begin
    self.ShowPopup(pt.x, pt.y);
end;
procedure TACPopupPrototype.ShowPopup(x, y: integer; NoSizeCalc : boolean = false);
var t : Cardinal;
    procedure SetNoActivate;
    begin
        SetWindowLong(self.Handle, GWL_EXSTYLE,
            GetWindowLong(self.Handle, GWL_EXSTYLE) or WS_EX_NOACTIVATE);
    end;
    procedure ClearNoActivate;
    begin
        SetWindowLong(self.Handle, GWL_EXSTYLE,
            GetWindowLong(self.Handle, GWL_EXSTYLE) and not WS_EX_NOACTIVATE);
    end;
    procedure HandleTaksbarIcon;
    begin
//        if (ShowOnTaskBar) then begin
//            if not FirstPaint then begin
//                SetWindowLong(
//                    self.Handle,
//                    GWL_EXSTYLE,
//                    GetWindowLong(Application.Handle, GWL_EXSTYLE) and not WS_EX_APPWINDOW or WS_EX_TOOLWINDOW
//                );
//            end else begin
//                if (fPopupMode.getMode <> pdmPinned) then begin
//                    SetWindowLong(
//                        self.Handle,
//                        GWL_EXSTYLE,
//                        GetWindowLong(Application.Handle, GWL_EXSTYLE) or WS_EX_APPWINDOW and not WS_EX_TOOLWINDOW
//                    );
//                end;
//            end;
//            ShowWindow(self.Handle, SW_HIDE);
//        end;
    end;
begin
	if fshowing then begin
    	FrmDebug.appendlog('ACPopup.ShowPopup called twice');
    	EXIT;
    end;

    if fItems.fSubMenu.count = 0 then Exit;
    windows.EnterCriticalSection(crit);
    lastActiveState := -1;
    try
        HandleTaksbarIcon;

        FrmDebug.StartTimer;
        fShowing := true;
        fIgnoreMouseMove := false;
        fIsTooTall := false;
        fLastKeyDown := 0;
        fToolTipNew.MaxWidth := fConfig.getTooltipMaxWidth;
        fToolTipNew.MaxHeight := fConfig.getTooltipMaxHeight;
        FrmDebug.AppendLog('acpopup start');



        timHover.Interval := fConfig.getTooltipDelayMS;
        CancelHovertimer;

        fMaxItemWidth := fConfig.getMenuMaxWidth;
        fShowExpandedShortcuts := false;



        fpoint.x := x;
        fpoint.y := y;
        self.Left := X;
        self.Top := Y;

        if Not NoSizeCalc then begin
            self.CalcTotalSize;
        end;
        self.fIgnoreKeypresses := true;



        Windows.SetWindowPos(
            self.Handle,HWND_TOPMOST,
            self.left,self.top,self.width,self.Height,
            //SWP_SHOWWINDOW or SWP_NOZORDER or SWP_NOACTIVATE
            SWP_SHOWWINDOW or SWP_NOACTIVATE
        );
        if fConfig.getNeedsFocus and fcanfocus then begin
            ClearNoActivate;
        end else begin
            SetNoActivate;
        end;
        self.visible := true;
        If fConfig.getNeedsFocus and fcanfocus then begin
            TFocusManager.ForceForeground(self.handle);
        end;

        //self.Paint;

        FrmDebug.EndTimerLog('acpopup end');
    finally
        if fitems.fsubmenu.count = 0 then begin
            FrmDebug.AppendLog('acpopup error - empty popup, hiding');
            self.fShowing := true;
            self.Hide;
        end;

        self.fIgnoreKeypresses := False;
        windows.LeaveCriticalSection(crit);
    end;
end;
procedure TACPopupPrototype.DodgePoint(pt : TPoint);
var m : TMonitor;
begin
    m := screen.MonitorFromPoint(pt);
    self.Top := pt.Y - 10 - self.Height;
    if (m <> nil) then begin
        if self.Top < m.Top then begin
            self.Top := m.Top;
        end;
    end;
end;

{Popuplate Menu}
procedure TACPopupPrototype.ClearMainKeystrokes;
begin
    fKeystrokes.Clear;
end;
function TACPopupPrototype.IsDoubleHeightType(p : TACPopupItem) : boolean;
begin
    result := false;
end;
procedure TACPopupPrototype.RebuildPopup(DoAutoPopulate : boolean=true);
var c : Cardinal;
begin
    if DoAutoPopulate then begin
        fKeystrokes.Clear;
        self.flastRebuildUsedKeyboard := self.UseKeyboard;
        self.AutoPopulate;
    end;
    self.CalcTotalSize;
    self.Invalidate;
end;
procedure TACPopupPrototype.DetectKeystroke(p: TACPopupItem;useUppercase : Boolean = true);
var s : string;
	c : char;
    i : integer;
begin
    // detect non-literal '&' accelerators

    s := p.Prefix + p.Caption;
    s := StringReplace(s,'&&','',[rfReplaceAll]);
    i := Pos('&', s);
    if i <> 0 then begin
        c := s[i+1];
        if useUppercase then begin
            c := upcase(c);
            p.fcaption := StringReplace(p.fcaption, '&'+c,'&'+c,[rfIgnoreCase]);
        end;
        fKeystrokes.AddObject(c, p);
    end;
end;
function TACPopupPrototype.GetAcceleratorReserved(i: integer) : Char;
begin
    Result := self.GetAccelerator(
        i - (ClipQueue.GetQueueSize+fReservedCount)
    );
end;


function TACPopupPrototype.GetAccelerator(i: integer; NoReserve : boolean = false): char;
var other : string;
	alphanumeric : string;
begin
    result := ' ';
    if NoReserve then begin

    end else begin
        inc(i, fAcceleratorCount+fReservedCount);
    end;

    if (i>TotalShortcutKeysAvailable) then
        EXIT;

    alphanumeric := fconfig.GetAccelAlphaNumeric;
    if (i<length(alphanumeric)) then begin
    	result := alphanumeric[i+1];
    end else begin
    	other := fConfig.GetAccelSymbols;
        if (i-length(alphanumeric)) < length(other) then begin {exclude the last character}
        	result := other[1+(i-length(alphanumeric))]; {start with 1 as an index}
        end;
    end;
end;


function TACPopupPrototype.AddBreak: TACPopupItem;
begin
    result := fItems.AddBreak;
end;
procedure TACPopupPrototype.AddLine;
begin
    AddLineToNext := true;
end;
procedure TACPopupPrototype.Reset;
var i, j : integer;
    ac, subac : TACPopupItem;
begin
    //
    // a Reset must not be called directly after a HIDE
    // callers may expect data to still exist - especially
    // the clicked ACPopupItem
    //

    // The Popup is responsible for clearing all of the
    // PopupItems when showing a new popup
    //

    for i := fitems.fSubMenu.Count - 1 downto 0 do begin
        ac := fitems.fSubMenu.Items[i];

        if ac.IsSubmenu then begin
            for j := ac.fSubMenu.Count - 1 downto 0 do begin
                subac := ac.fSubMenu.Items[j];
                subac.Free;
            end;
        end;

        ac.Free;
    end;
    fitems.fSubMenu.Clear;

    for i := ftools.fSubMenu.Count - 1 downto 0 do begin
        ac := ftools.fSubMenu.Items[i];
        ac.Free;
    end;
    ftools.fSubmenu.clear;


    fKeystrokes.Clear;
    fExpandedKeystrokes.Clear;
    ClearHover;
    fLastVisible := nil;
    fLastVisible := nil;
    self.fLastSubmenu := nil;
    self.fQueuedSubmenu := nil;
    self.fMouseDownOn := nil;
    self.fDragMenuItem := nil;
    self.fDropMenuItem := nil;
end;

function TACPopupPrototype.isTool(p : TACPopupItem) : boolean;
begin
    result := not (fTools.SubMenu.IndexOf(p) =-1);
end;

{utility routines}
function TACPopupPrototype.GetItemAt(x, y: integer): TACPopupItem;
var i : integer;
    p : TACPopupItem;

    function CheckSubmenus(p : TACPopupItem) : TACPopupItem;
    var i : integer;
        sub : TACPopupItem;
    begin
        result := nil;
        if p = nil then EXIT;

        if (p.fSubShowing <> nil) then begin
            // check nested submenu first
            result := CheckSubmenus(p.fSubShowing);
            if result <> nil then EXIT;
        end;
        for i := 0 to p.fsubmenu.count-1 do begin
            sub := p.fSubMenu[i];

            if (x >= sub.fLeft) and (x <= sub.fRight) then begin
                if (y <= sub.fBottom) and (y >= sub.fTop) then begin
                    result := sub;
                    EXIT;
                end;
            end;
        end;
    end;
begin
    // check submenus first
    result := CheckSubmenus(fitems.fSubShowing);
    if result <> nil then EXIT;

    for i := 0 to fitems.fsubmenu.count-1 do begin
        p := fitems.fsubmenu.Items[i];
        if not p.Visible then begin
            p := nil;
            Continue;
        end;
        {same column}
        if (x >= p.fLeft) and (x <= p.fRight) then begin
            if (y <= p.fBottom) and (y>=p.fTop) then begin
            	if p.isbreak then EXIT;

                result := p;
                EXIT;
            end;
        end;
        p := nil;
    end;

    if (p=nil) and (result=nil) then begin
        for i := 0 to ftools.fSubMenu.Count-1 do begin
            p := ftools.fSubMenu.Items[i];
            if (x >= p.fLeft) and (x <= p.fRight) then begin
                if (y <= p.fBottom) and (y>=p.fTop) then begin
                    if p.isbreak then EXIT;

                    result := p;
                    EXIT;
                end;
            end;
        end;
    end;
end;
function TACPopupPrototype.MouseInPopup : boolean;
var r : TRect;
begin
    r := Rect(self.left, self.Top,self.Left+self.Width,self.Top+self.Height);

    result := PtInRect(r, mouse.CursorPos);
    if not result then begin
        if fitems.fSubShowing <> nil then begin
            r := Rect(
                fitems.fSubShowing.Left,fitems.fSubShowing.Top,
                fitems.fSubShowing.right, fitems.fSubShowing.bottom);
            result := PtInRect(r, mouse.CursorPos);
        end;
    end;
end;
procedure TACPopupPrototype.AssignSubmenuShortcuts(p : TACPopupItem);
var i,j : integer;
begin
    if self.fUseExpandedShortcuts and  self.fKeyActivated and fShowExpandedShortcuts then begin
        if p.IsSubmenu or p.IsExpandable then begin
            j := 0;
            fExpandedKeystrokes.Clear;
            for i := 0 to p.fSubMenu.Count - 1 do begin
                with p.fSubMenu.Items[i] do begin
                    if not IsLine and not IsGap then begin
                        Prefix := '&' +self.GetAcceleratorReserved(j);
                        fExpandedKeystrokes.AddObject(Prefix[2], p.fSubMenu.Items[i]);
                        Inc(j);
                    end;
                end;
            end;
        end;
    end else begin
        if p.IsSubmenu or p.IsExpandable then begin
            for i := 0 to p.fSubMenu.Count - 1 do begin
                with p.fSubMenu.Items[i] do begin
                    if not IsLine and not IsGap then begin
                        Prefix := '';
                    end;
                end;
            end;
        end;
    end;
end;
procedure TACPopupPrototype.SetShowExpandedShortcuts(value : Boolean);
begin
    fShowExpandedShortcuts := value;
end;
procedure TACPopupPrototype.SetHoverUnderMouse;
var pt : TPoint;
    ac : TACPopupItem;
begin
    Application.ProcessMessages;
    pt := ScreenToClient(mouse.CursorPos);
    ac := self.GetItemAt(pt.X,pt.y);
    SetHover(ac, false);
end;
function TACPopupPrototype.UseKeyboard : boolean;
var
    isForeground : boolean;
begin
    isForeground := (GetForegroundWindow = self.Handle);
    result := (fconfig.getUseKeyboard and (fConfig.getNeedsFocus));

    if (fPopupMode.equals(pdmPinned)) then result := isForeground;
end;
function TACPopupPrototype.UsePrefix : boolean;
begin
    result := UseKeyboard or (fPopupMode.equals(pdmPinned));
end;
procedure TACPopupPrototype.SetFormMode(value : boolean);
begin
    if value then begin
        fPopupMode.setMode(pdmFormMode);
    end else begin
        fPopupMode.setMode(pdmAutoHide);
    end;
end;
function TACPopupPrototype.GetFormMode : boolean;
begin
    result := fPopupMode.equals(pdmFormMode);
end;



{Drawing Routines}
procedure TACPopupPrototype.SetCanvas(c: TCanvas);
begin
    fNewCanvas := c;
end;
function TACPopupPrototype.GetCanvas: TCanvas;
begin
    if fNewCanvas = nil then begin
    	result := self.Canvas;
    end else begin
        result := fNewCanvas;
    end;
end;

procedure TACPopupPrototype.CalcTotalSize;
    function CalcRect(p : TACPopupItem; top,left : integer; isMainPopup : boolean = false) : TRect;
    const GAP_SIZE = 20;
    var wd,ht,i,col, leftoffset, maxwd, totalht, totalwd, maxht, originaltop, y : integer;
        sub : TACPopupItem;
        pop : TACPopupItem;
        procedure TallyColumn(wd : Integer);
        begin
            FrmDebug.AppendLog('New Column, old wd = ' + inttostr(wd));
            p.fColWidth[col] := wd;
            inc(col);
        end;
        procedure CalcItem(sub : TACPopupItem; top,left : Integer;  var wd : integer; var ht : integer);
        var
            drawablePic : boolean;
            IsPic : boolean;
            isWide : boolean;
            i : integer;
        begin
            wd := 0; ht := 0;
            sub.ftop := top;


            if sub.IsLine then begin
                Inc(ht, BLANK_SPACING + 5);
            end else begin
                wd := CalcWidth(sub, false);
                isWide := ((wd+getLeftSpaceWidth(p)) > trunc(fMaxItemWidth*1.3));
                wd := min(self.fMaxItemWidth, wd);

                IsPic := (sub.IsPictureClip);
                drawablePic := IsPic and
                    (fConfig.getUseThumbnails);


                if IsDoubleHeightType(sub) and
                    (drawablePic or ((not IsPic) and  fConfig.getUseDoubleHeight and isWide)) then begin
                    sub.DoubleHeight := true;
                    ht :=  CalcHeight(sub);
                end else begin
                    sub.DoubleHeight := false;
                    ht := CalcHeight(sub); //adding spacing screws up the aesthetics
                end;

                if (sub.DoubleHeight) then begin
                    if (sub.IsPictureClip) and
                        (sub.Clip.GetPicHeight < ht)then begin
                        sub.DoubleHeight := false;
                        ht := CalcHeight(sub); //adding spacing screws up the aesthetics
                    end;
                end;
            end;

            sub.fBottom := top+ht-1; // coordiants are zero based
            sub.fleft := left;
            sub.fRight := left+wd;
            sub.fMyColumn := col;
        end;
    begin
        totalht := 0;
        inc(totalht,POPUP_TOP_BORDER);
        if (ShowCaption) and (isMainPopup) then begin
            inc(totalht, CaptionHeight);
        end;

        originaltop := top;
        col := 0;
        maxwd := 0; maxht := 0;
        totalwd := 0;
        leftoffset := 0;
        p.fColWidth[col] := 0;

        for i := 0 to p.fSubMenu.count - 1 do begin
            sub := p.fSubMenu.Items[i];
            sub.fMyColumn := col;
            if sub.IsBreak then begin
                // break also doubles as a blank space when the second column
                // is taller
                sub.ftop := top+totalht;
                sub.fBottom := sub.fTop;
                sub.fleft := left;
                sub.fMyColumn := col;

                leftoffset := maxwd;
                TallyColumn(maxwd);
                inc(totalht, POPUP_BOTTOM_BORDER);
                maxht := max(totalht, maxht);
                Inc(totalwd, maxwd);
                totalht := 0;
                inc(totalht,POPUP_TOP_BORDER);
                if (ShowCaption) then begin
                    inc(totalht, CaptionHeight);
                end;
                maxwd := 0;
            end else if sub.Visible then begin
                CalcItem(sub, top+totalht, left + leftoffset, wd, ht);
                Inc(totalht, ht);
                maxwd := max(wd, maxwd);
            end;
        end;



        TallyColumn(maxwd);
        inc(totalwd, maxwd);
        maxwd := 0;
        Inc(totalht, POPUP_BOTTOM_BORDER);
        maxht := max(maxht, totalht);

        for i := 0 to p.fSubMenu.Count-1 do begin
            sub := p.fSubMenu.Items[i];
            //if not sub.Visible then CONTINUE;

            sub.fRight :=  sub.fLeft + p.fColWidth[sub.ColumnIndex];
        end;

        result.top := originaltop;
        result.Left := left;
        result.Bottom := originaltop + maxht;
        result.Right := left + (totalwd);

        result.Width := result.Width + POPUP_LEFT_BORDER+POPUP_RIGHT_BORDER;
        p.fMaxHeight := maxht;

        if isMainPopup then begin
            fMainRect.Top := result.Top;
            fMainRect.left := result.Left;
            fMainRect.Right := result.Right;
            fMainRect.bottom := result.bottom;
        end;

    end;
    procedure CalcTools(p : TACPopupItem; top,left : integer);
    var
        leftoffset : integer;
        rightoffset : integer;
        procedure CalcItem(sub : TACPopupItem);
        var
            twd, itemht : integer;
            drawablePic : boolean;
            IsPic : boolean;
            isWide : boolean;
        begin
            sub.ftop := top + POPUP_TOP_BORDER;
            twd := CalcToolWidth(sub);
            //itemht := CalcHeight(sub);
            itemht := POPUP_TOP_BORDER+CaptionHeight-POPUP_BOTTOM_BORDER-BLANK_SPACING;

            sub.fBottom := sub.ftop + itemht;
            if (not leftToolsList.Contains(sub)) then begin
                sub.fLeft := rightoffset-twd;
                dec(rightoffset,twd);
            end else begin
                sub.fleft := leftoffset;
                inc(leftoffset, twd);
            end;

            sub.fStaticRight := true;
            sub.fRight := sub.fleft + twd;
        end;
    var
        i : integer;
        sub : TACPopupItem;
    begin
        leftoffset := 0;
        rightoffset := fItems.fSubRect.Right - POPUP_RIGHT_BORDER-1;
        for i := 0 to p.fSubMenu.count - 1 do begin
            sub := p.fSubMenu.Items[i];
            CalcItem(sub);
        end;
    end;


    procedure FixMainVisibility(p : TACPopupItem; r : TRect; addGap : boolean);
    var
        offset : integer;
        m : TMonitor;
    begin
        offset := fPoint.Y;
        if (offset + r.top + r.Height > screen.DesktopRect.Bottom) then begin
            if (r.Height > screen.DesktopRect.Bottom) then begin
                offset := 0;
                top := 0;
            end else begin
                offset := ((offset + r.top + r.Height) - screen.desktoprect.bottom);
            end;
            self.Top := top - offset;
            fTopFixed := true;
            fIsTooTall := true;
        end;
        offset := fPoint.X;
        if (offset + r.Left+r.Width > screen.DesktopRect.right) then begin
            if (r.Width > Screen.DesktopRect.Right) then begin
                offset := 0;
                left := 0;
            end else begin
                offset := (offset + r.Left+r.Width) - screen.DesktopRect.right;
            end;

            self.Left := Self.Left - offset;
            fLeftFixed := true;
        end;

        m := screen.MonitorFromWindow(self.Handle);
        if (m <> nil) then begin
            if (self.top+r.Height) > m.BoundsRect.Bottom then begin
                fIsTooTall := true;
            end;
        end;
    end;
    function FixVisibily(p : TACPopupItem; r : TRect; addGap : boolean) : TRect;
    var
        xoffset, yoffset, ht : integer;
        pop : TACPopupItem;
        i : integer;
    begin
        result := r;
        yoffset := fPoint.Y;
        xoffset := fPoint.X;
        if (yoffset + r.top + r.Height > screen.DesktopRect.Bottom) then begin
            yoffset := ((yoffset + r.top + r.Height) - screen.desktoprect.bottom);
        end else begin
            yoffset := 0;
        end;

        if (xoffset + r.Left+r.Width > screen.DesktopRect.right) then begin
            if addGap then
                if (p.fSubMenu.Count <> 0) and
                (not p.fSubMenu.Items[0].isGap) then begin
                    pop := p.add;
                    pop.SetIsGap(true);
                    p.fSubMenu.Move(p.fSubMenu.Count-1, 0);
                end;
            xoffset := (xoffset + r.Left+r.Width) - screen.DesktopRect.right;
        end else begin
            xoffset := 0;
        end;

        if (yoffset <> 0) or (xoffset<>0) then begin
            result := CalcRect(
                p,
                max(0,r.Top-yoffset),
               (r.Left - xoffset)
            );
        end;

    end;
var
    r, subr : TRect;
    p : TACPopupItem;
    i : integer;
    top : integer;
    wd : integer;
    h : THandle;
begin
try
    GetCanvas.Lock;
    h := GetCanvas.Handle;

    FrmDebug.AppendLog('calcsize start');
    FrmDebug.AppendLog('calcsize locks = '+IntToStr(GetCanvas.LockCount));
    region1 := 0;
    top := 0;

    toolsMinWidth := 0;
    for i := 0 to ftools.fSubMenu.Count-1 do begin
        inc(toolsMinWidth, CalcToolWidth(ftools.fSubMenu.Items[i]));
    end;


    r := CalcRect(fitems, top, 0, true);
    CopyRect(fItems.fSubRect, r);
    FixMainVisibility(fItems, r, false);
    if System.Odd(r.Width) then begin
        r.Width := r.Width + 1;
    end;
    CopyRect(fItems.fSubRect, r);

    if CheckWin32Version(6, 0) and true then begin
        r.Width := r.width + 1;
        region1 := Windows.CreateRectRgnIndirect(r);
        r.Width := r.Width -1;
    end;

    if fitems.fSubShowing <> nil then begin
        subr := CalcRect(fitems.fSubShowing,fitems.fSubShowing.ftop, fitems.fSubShowing.right+1);
        CopyRect(fItems.fSubShowing.fSubRect, subr);
        subr := FixVisibily(fitems.fSubShowing, subr, true);

        if System.Odd(r.Width) then begin
            r.Width := r.Width + 1;
        end;

        CopyRect(fItems.fSubShowing.fSubRect, subr);

        if CheckWin32Version(6, 0) and true then begin
            subr.Width := subr.width + 1;
            region2 := Windows.CreateRectRgnIndirect(subr);
            windows.CombineRgn(region1,region1,region2, RGN_OR);
            Windows.DeleteObject(region2);
            subr.Width := subr.Width -1;
        end;

        r := System.Types.UnionRect(r, subr);
    end;

    //TODO: learn why an odd width is causing a visual artifact
    // for the window's shadow
    // tested: it's not the CombineRgn
    if System.Odd(r.Width) then begin
        r.Width := r.Width + 1;
    end;
    self.Width := r.Width+1;
    self.Height := r.height;


    FrmDebug.AppendLog('Find the last visible item');
    i := fitems.fSubMenu.count-1;
    if i >= 0 then begin
        repeat
            p := fitems.fSubMenu.GetItem(i);
            dec(i);
        until (not p.IsBreak and p.Visible) or (i=0);
    end;
    fLastVisible := p;

    if ShowCaption then begin
        calcTools(self.fTools, 0, 0);
    end;

    wd := 0;
    for i:= 0 to ftools.fSubMenu.Count-1 do begin
        inc(wd, (ftools.fSubMenu.Items[i].left -ftools.fSubMenu.Items[i].left) );
    end;
    toolsMinWidth := wd;


    if CheckWin32Version(6, 0) and fShowShadow then begin
		Windows.SetWindowRgn(self.Handle,region1,false);
         // Windows owns this handle now
    end;
    FrmDebug.AppendLog('calcsize end');
    FrmDebug.AppendLog('Canvas same = ' + BoolToStr(GetCanvas.Handle = h));
    PostSizeCalculationEvent;
finally
    GetCanvas.Unlock;
    FrmDebug.AppendLog('calcsize locks = '+IntToStr(GetCanvas.LockCount));
end;
end;
function TACPopupPrototype.CalcHeight(fpi: TACPopupItem): integer;
var i : integer;
    s : string;
    tempr : TRect;
    showSmall : boolean;
begin
    GetCanvas.Lock;
    result := max(
        GetCanvas.TextHeight('AZW')+HEIGHT_PADDING,
        fConfig.getMenuMinimumHeight
    );

    showSmall := fpi.SmallCaption and not (fpi.NoSmallHeight);
    i := GetCanvas.Font.Size;
    GetCanvas.Font.Size := i-1;
    if  showSmall then begin
        result := GetCanvas.TextHeight('AZW')+HEIGHT_PADDING;
    end else begin
        GetCanvas.Font.Size := i;
    end;

    if fpi.DoubleHeight then begin
        s := 'AZ'#13#10'AZ';
        DrawTextW(GetCanvas.Handle, pwidechar(s), length(s),
            tempr, DT_CALCRECT);
        result := max(result,tempr.Bottom-tempr.Top) +HEIGHT_PADDING
    end;
    GetCanvas.Font.Size := i;

    if fpi.SeparatorLine then inc(result,LINE_HEIGHT);

    GetCanvas.Unlock;
end;
function TACPopupPrototype.CalcCaptionHeight : integer;
const
    MENU_BLANK_SPACE = BLANK_SPACING + HEIGHT_PADDING;
begin
    GetCanvas.Lock;
    result :=  max(ICON_HEIGHT, GetCanvas.TextHeight('AZ')) + MENU_BLANK_SPACE ;
    result := max(result, Config.getMenuMinimumHeight+MENU_BLANK_SPACE);
    GetCanvas.Unlock;
end;
function TACPopupPrototype.CalcWidth(fpi: TACPopupItem; ClipIt : boolean=true; MinWidth : boolean=true): integer;
var
    s : string;
    i : integer;
const
    MIN_WIDTH = 120;
begin
    GetCanvas.Lock;
    result := POPUP_LEFT_BORDER + getLeftSpaceWidth(fpi) + fpi.DesiredWidth(self, GetCanvas) +POPUP_RIGHT_BORDER;
    if fpi.IsExpandable or fpi.IsSubmenu then begin
        inc(result, RIGHT_SPACE);
    end;

    if ClipIt then begin
    	result := min(result, self.fMaxItemWidth);
    end;
    if (MinWidth) then begin
        if ShowCaption then begin
            result := max(result, toolsMinWidth);
        end;
        result := max(MIN_WIDTH, result);
    end;
    GetCanvas.Unlock;
end;
function TACPopupPrototype.CalcToolWidth(pi : TACPopupItem) : integer;
begin
    result := CalcWidth(pi, false, false);
    if  (pi.LeftIcon.Bitmap<>nil) and (pi.LeftIcon.Bitmap.Width <> ICON_WIDTH) then begin
        dec(result, ICON_WIDTH);
        inc(result, pi.LeftIcon.Bitmap.Width);
    end;
    if (pi.LeftIcon.Bitmap = nil) then begin
        dec(result, ICON_WIDTH);
    end;
end;
procedure TACPopupPrototype.PostSizeCalculationEvent;
begin
    // purposely blank
end;
function TACPopupPrototype.getLeftSpaceWidth(p : TACPopupItem) : integer;
begin
   result := (MENU_CAPTION_LEFT);
end;
function TACPopupPrototype.CalcIconRect(p : TACPopupItem; bounds : TRect) : TRect;
var
    ht : integer;
    wd : integer;
begin
    result := CalcIconBackgroundRect(p, bounds);
    ht := result.Height;
    wd := result.Width;
    if (ht > ICON_HEIGHT + BLANK_SPACING) then begin
        result.Top := result.top + (ht - ICON_HEIGHT) div 2;
        result.Left := result.left + (wd - ICON_WIDTH) div 2;
        Result.Height := ICON_HEIGHT + BLANK_SPACING;
        //result.Width := ICON_WIDTH + FSPACING;
    end else begin
        Result.Top := result.Top + (BLANK_SPACING div 2);
    end;
    if p.LeftIcon.CliptypeIcon <> nil then begin
        result.left := result.left - (BLANK_SPACING div 2);
        Result.Width := result.Width + (BLANK_SPACING div 2);
    end;
end;
function TACPopupPrototype.CalcIconBackgroundRect(p : TACPopupItem; bounds : TRect) : TRect;
begin
    Result.Top := bounds.Top;
    Result.Left := bounds.Left;
    Result.Right := Result.Left + getLeftSpaceWidth(p);
    Result.Bottom := bounds.Bottom;
end;

procedure TACPopupPrototype.FullRedraw;
    procedure DrawColumns;
    var
        i : integer;
        r : TRect;
        offset : Integer;
    begin
        // columns can have blank spaces, since they won't be the same
        // height most of the time
        offset := 0;
        for i := 0 to fitems.fColWidth.Count-1 do begin
            CopyRect(r, fItems.fSubRect);
            r.Left := r.Left + offset;
            r.Right := r.Left + (getLeftSpaceWidth(nil)-1)+MENU_LEFT_EDGE+1;


            GetCanvas.Brush.color := fColumnColor;

            //testing myfillrect
            //UseCanvas.FillRect(r);
            MyFillRect(r);
            GetCanvas.Pen.Color :=  fBorderColor;
            GetCanvas.MoveTo(r.right,r.top);
            GetCanvas.LineTo(r.right,r.bottom);
            GetCanvas.MoveTo(r.left,r.top);
            GetCanvas.LineTo(r.left,r.bottom);

            Inc(offset, fitems.fColWidth[i]);
        end;
    end;
    procedure DrawMenu(p : TACPopupItem; const top,left : integer);
    var i, first, last : integer;
        sub, sub2 : TACPopupItem;
        r : TRect;
    begin
        if p.fSubMenu.Count = 0 then EXIT;
        CopyRect(r, p.fSubRect);

        GetCanvas.Pen.Color := fBackgroundColor;
        GetCanvas.Brush.color := fBackgroundColor;

        MyFillRect(r);
        if (p=fItems) then begin
            DrawColumns;
        end;


        // draw individual items
        for i := 0 to p.fsubmenu.count - 1 do begin
            if not p.fsubmenu.Items[i].Visible then Continue;
            sub := p.fsubmenu.Items[i];

            if sub.IsBreak  then begin
                // handle blank space
                if r.bottom - sub.fBottom > POPUP_BOTTOM_BORDER then begin
                    sub.fBottom := r.Bottom;
                    self.DrawItem(sub);
                end;
            end else begin
                self.DrawItem(sub);
            end;
        end;

        {top line}
        GetCanvas.pen.color := fBorderColor;
        GetCanvas.Pen.Width := 1;
        GetCanvas.MoveTo(r.left, r.top);
        GetCanvas.lineto(r.right, r.top);
        {bottom line}
        GetCanvas.pen.color := fBorderColor;
        GetCanvas.MoveTo(r.left, r.Bottom-1);
        GetCanvas.lineto(r.right, r.Bottom-1);
        {right line}
        GetCanvas.pen.color := fBorderColor;
        GetCanvas.MoveTo(r.right, r.top);
        GetCanvas.lineto(r.right, r.bottom);

        // handle blank space


        if p.fSubShowing <> nil then begin
            DrawMenu(p.fSubShowing, p.fSubShowing.fTop, p.fSubShowing.Right+1);
        end;
    end;
    procedure DrawTools(p : TACPopupItem; const top,left : integer);
    var
        i : integer;
        sub : TACPopupItem;
        r : TRect;
        procedure leftline;
        begin
            GetCanvas.pen.color := fBorderColor;
            GetCanvas.MoveTo(left, r.top);
            GetCanvas.lineto(left, r.bottom+1);
        end;
        procedure bottomline;
        begin
            GetCanvas.pen.color := BlendOrReverse(fCaptionColor, clblack, clwhite, 93);
            GetCanvas.MoveTo(r.left+1, r.bottom-1);
            GetCanvas.lineto(r.right, r.bottom-1);
        end;
    begin
        r.Top := top+POPUP_TOP_BORDER;
        r.Left := left+POPUP_LEFT_BORDER;
        r.Right := fItems.fSubRect.Right-POPUP_RIGHT_BORDER-1;
        r.Bottom := r.Top+ CaptionHeight;
        GetCanvas.Brush.Color := fCaptionColor;
        //UseCanvas.FillRect(r);
        MyFillRect(r);

        for i := 0 to p.fSubMenu.Count-1 do begin
            sub := p.fSubMenu.Items[i];
            self.DrawItem(sub);
        end;

        leftline;
        bottomline;
    end;
begin
    GetCanvas.Lock;
    FrmDebug.AppendLog('FullRedraw locks = '+IntToStr(GetCanvas.LockCount));
    drawmenu(fitems, 0,0);
    if ShowCaption then begin
        drawtools(ftools, 0,0);
        if (false) then begin
            GetCanvas.pen.color := fColumnEdgeColor;
            GetCanvas.Pen.Width := 1;
            GetCanvas.MoveTo(0, CaptionHeight);
            GetCanvas.lineto(fItems.fSubRect.right-POPUP_RIGHT_BORDER, CaptionHeight);

            GetCanvas.pen.color := fColumnEdgeColor;
            GetCanvas.Pen.Width := 1;
            GetCanvas.MoveTo(0, 0);
            GetCanvas.lineto(0, 0 + CaptionHeight);
        end;
    end;

    fFirstPaint := true;
    DrawDragmark;

    if self.GetFormMode then begin
        SetWindowLong(self.Handle,GWL_EXSTYLE, GetWindowLong(self.handle, GWL_EXSTYLE) or WS_EX_LAYERED);
        SetLayeredWindowAttributes(self.handle,0,trunc(255 * 0.80), LWA_ALPHA);
    end else begin
		SetWindowLong(self.Handle,GWL_EXSTYLE, GetWindowLong(self.handle, GWL_EXSTYLE) and not WS_EX_LAYERED);
    end;
    GetCanvas.Unlock;
    FrmDebug.AppendLog('FullRedraw locks = '+IntToStr(GetCanvas.LockCount));
end;

procedure TACPopupPrototype.MyFillRect(r : TRect);
begin
    GetCanvas.Lock;
    inc(r.bottom);
    inc(r.Right);
    GetCanvas.FillRect(r);
    GetCanvas.Unlock;
end;
procedure TACPopupPrototype.DrawItem(p: TACPopupItem);
var r : Trect;
    wc : TWideChar;
    s : string;
    len: integer;
    c : TColor;

    mainr : TRect;
    iconr : TRect;
    sd : TScrollDirection;
    prefixwidth : integer;
    clipboardleft : integer;
    programname : string;
const
    BLEND_CLICKED = 97;
    BLEND_CLICKED_ORIGINAL = 99;
    DESATURATE_PERCENT = 60;
    procedure HandleBottomBorder;
        function IsLastInExpandable(p : TACPopupItem) : boolean;
        begin
            result := p.IsInExpandedMenu and (
                p.fParent.fSubMenu.Items[p.fParent.fSubMenu.Count-1] = p
            );
        end;
    begin
        if (p.IsExpandable and p.Expanded) or IsLastInExpandable(p) then begin
            GetCanvas.Pen.Color := DimColor(fColumnColor, 0.90);
            GetCanvas.MoveTo(mainr.Left, mainr.Bottom);
            GetCanvas.LineTo(mainr.Right,mainr.Bottom);
        end;
    end;
    function CaptionLeft : integer;
    begin
        result := mainr.left + getLeftSpaceWidth(p) + MENU_LEFT_EDGE - 1;
    end;

    procedure MyGradientFill(c1, c2 : TColor; r : TRect; d : TGradientDirection);
    begin
        inc(r.Bottom);
        inc(r.Right);
        GraphUtil.GradientFillCanvas(GetCanvas,c1,c2,r,d);
    end;
    procedure VerticalLines;
    begin
        // outside boundary line
        // reserve this space
        CopyRect(r,mainr);
        inc(r.Bottom);

        GetCanvas.pen.Color := fBorderColor;
        GetCanvas.MoveTo(r.Left,r.top);
        GetCanvas.LineTo(r.left,r.bottom+1);
        inc(r.Left);
        GetCanvas.pen.Color := dimColor(fBackgroundColor,1.02);
        GetCanvas.MoveTo(r.Left,r.top);
        GetCanvas.LineTo(r.left,r.bottom+1);
    end;
    procedure LeftIconBackground(nohover : boolean=false);
        function ClickableHoveredIcon : boolean;
        begin
            result := (not nohover) and(hover = p) and
                (p.ClickableLeftIcon);
        end;
    var r : trect;
    begin
        {icon column}
        CopyRect(r,mainr);
        r := CalcIconBackgroundRect(p, mainr);
        //r.Right := CaptionLeft;
        GetCanvas.brush.Style := bsSolid;
        GetCanvas.Brush.Color := fColumnColor;
        if p.IsInExpandedMenu then begin
            GetCanvas.Brush.Color := fExpandedBackgroundColor;
        end;
        if (fTools.SubMenu.IndexOf(p) <> -1) then begin
            GetCanvas.Brush.Color := fCaptionColor;
        end;
        if ClickableHoveredIcon and not p.IsExpandable then begin
            GetCanvas.Brush.Color := Blend(GetCanvas.Brush.Color,fFontColor,93);
        end;
        MyFillRect(r);

        {focus rectangle}
        if ClickableHoveredIcon then begin
            if FLastPosition = pipLeftIcon then begin
                GetCanvas.Pen.Color := fFontColor;
                GetCanvas.Pen.style := psDot;
                GetCanvas.Rectangle(r);
                GetCanvas.Pen.style := psSolid;
            end else begin
                GetCanvas.Pen.Color := BlendOrReverse(GetCanvas.Brush.Color,clWhite,clBlack, 70);
                GetCanvas.MoveTo(r.Left, r.top);
                GetCanvas.LineTo(r.Right, r.top);

                GetCanvas.Pen.Color := BlendOrReverse(GetCanvas.Brush.Color,clBlack,clWhite, 75);
                GetCanvas.MoveTo(r.Left, r.bottom);
                GetCanvas.LineTo(r.Right, r.bottom);
            end;
        end;
    end;
    procedure CaptionBackground(forceplain : boolean = false);
    var c1, c2 : TColor;
        liner : TRect;
        procedure ExpandedBoxing;
        begin
           if p.Expanded then begin
                c1 := fBackgroundBase;
                c2 := BlendOrReverse(c1, clBlack,clWhite, 90);

                GetCanvas.brush.Color := c1;
                MyFillRect(r);

                copyrect(liner, r);
                liner.Left := liner.right - RIGHT_MENU_GRADIENT_WIDTH;
                MyGradientFill(c1,c2,liner,gdHorizontal);


            end else begin
                c1 := fBackgroundBase;
                GetCanvas.brush.Color := c1;
                MyFillRect(r);

                c2 := BlendOrReverse(c1, clBlack,clWhite, 90);
                copyrect(liner, r);
                liner.Left := liner.right - RIGHT_MENU_GRADIENT_WIDTH;
                MyGradientFill(c1,c2,liner,gdHorizontal);

                // top line
//              UseCanvas.Pen.Color := blend(fBackgroundColor,clWhite, 85);
//                UseCanvas.MoveTo(r.Left, r.top);
//                UseCanvas.LineTo(r.Right, r.top);
//                UseCanvas.MoveTo(r.Left, r.top);
//                UseCanvas.LineTo(r.Left, r.bottom);

                //bottom line

//                UseCanvas.Pen.Color := blend(fBackgroundColor,clBlack, 90);
//                UseCanvas.MoveTo(r.Left, r.bottom);
//                UseCanvas.LineTo(r.Right, r.bottom);
//                UseCanvas.MoveTo(r.Right-1, r.top);
//                UseCanvas.LineTo(r.Right-1, r.bottom);
            end;
        end;
    begin
        CopyRect(r,mainr);
        r.Left := CaptionLeft+1;
        r.right := r.right;
        GetCanvas.Brush.Style := bsSolid;

        GetCanvas.brush.Color := fBackgroundColor;
        MyFillRect(r);
        fbackgroundBase := 0;

        if forceplain then begin
            fBackgroundBase := GetCanvas.Brush.Color;
            p.DrawCaptionBackground(self,GetCanvas,r);
            EXIT;
        end;

        if p.IsExpandable then begin
            fbackgroundBase := fColumnColor;
        end else begin
            fbackgroundBase := fBackgroundColor;
        end;

        if p.IsExpandable then begin
            ExpandedBoxing;
        end else begin
            p.DrawCaptionBackground(self, GetCanvas, r);
        end;

        if p.BottomLine then begin
            GetCanvas.Pen.Color := blend(fFontColor, fBackgroundBase, 18);
            GetCanvas.MoveTo(r.Left+6,r.bottom-1);
            GetCanvas.LineTo(r.Right-6,r.bottom-1);
        end;
    end;
    procedure LeftColumnEdge;
    begin
        CopyRect(r, mainr);
        inc(r.bottom); // off-by-one drawing
        r.Right := CaptionLeft;
        GetCanvas.pen.Color := fColumnEdgeColor;
        if p.IsInExpandedMenu then begin
            GetCanvas.pen.Color := BlendOrReverse(fBackgroundColor, fColumnEdgeColor,clWhite, 90);
        end;
        GetCanvas.MoveTo(r.right,r.top);
        GetCanvas.LineTo(r.right,r.bottom);
    end;
    procedure SeparatorLine;
    var y : integer;
    begin
        if p.SeparatorLine then begin
            y := r.Top + ((LINE_HEIGHT-2) div 2);
            LeftIconBackground(true);
            LeftColumnEdge;
            CaptionBackground(true);


            GetCanvas.Pen.Color := blend(fBackgroundBase,clwhite, 70);
            GetCanvas.MoveTo(r.Left+6,y);
            GetCanvas.LineTo(r.Right-6,y);

            GetCanvas.Pen.Color := blend(fBackgroundBase,clGray,  70);
            GetCanvas.MoveTo(r.Left+6,y+1);
            GetCanvas.LineTo(r.Right-6,y+1);

            inc(mainr.Top,LINE_HEIGHT);
        end;
    end;
    procedure DrawLine;
    begin
        {edge in between the rows}
        Copyrect(r, mainr);
        dec(r.Right);
        r.Left := CaptionLeft;
        r.Top := p.fTop + BLANK_SPACING+1;
        r.Bottom := r.top+1;
        GetCanvas.Brush.Color := 0;
        GetCanvas.pen.Color := dimColor(fBackgroundColor,0.95);
        GetCanvas.MoveTo(r.Left, r.top);
        GetCanvas.LineTo(r.Right, r.top);
        GetCanvas.pen.Color := dimColor(fBackgroundColor,1.10);
        GetCanvas.MoveTo(r.Left, r.Bottom);
        GetCanvas.LineTo(r.Right, r.bottom);
    end;
    procedure HanldeDoubleHeightMarker;
    var
        halfwd : integer;
        quarterwd : integer;
        box : TRect;
        c : TColor;
    begin
        if p.DoubleHeight then begin
            halfwd := (r.Right - r.Left) div 2;
            quarterwd := halfwd div 2;

            CopyRect(box, r);
            //box.left := box.right - quarterwd;

            c := BlendOrReverse(
                fBackgroundColor,
                Blend(clblack,clWhite,90),
                Blend(clwhite,clBlack,10),
                98);
            GetCanvas.pen.Color := c;
            GetCanvas.MoveTo(box.Left, box.Top);
            GetCanvas.LineTo(box.right, box.Top);
            GetCanvas.MoveTo(box.Left, box.bottom);
            GetCanvas.LineTo(box.right, box.bottom);
        end;
    end;
    procedure DrawHoverHilite;
    var startc : TColor;
        back : TColor;
        r : TRect;
        procedure Outline;
        begin
            c := fHighlightColor;
            c := Blend(c, fFontColor, 95);
            GetCanvas.pen.Color := UnitMisc.Blend(  fHighlightColor, fFontColor, 92);
            GetCanvas.RoundRect(
                r.left,r.top,r.Right,r.bottom,
                2,2
            );
        end;
    begin
        // Since it's only called once, there's no need to optimized the
        // colors

        copyrect(r,mainr);
        if (fTools.SubMenu.IndexOf(p) = -1) then begin
            r.Left := CaptionLeft;
        end;
        dec(r.right, POPUP_RIGHT_BORDER);

        back := fBackgroundColor;
        if p.IsInExpandedMenu then begin
            back := DimColorOrReverse(fBackgroundColor, 1.04);
        end;
        back := unitMisc.Blend(back, fHighlightColor, 70);   // was 90% background


        {focus rectangle}

        {inner gradient}
        //inc(r.Top);
        inc(r.left);
        inc(r.right);
        Inc(r.bottom);
        if (p.Disabled) then begin
            startc :=  UnitMisc.blend(fBackgroundBase, fDisabledColor, 90);
        end else begin
            startc := UnitMisc.Blend(
                fBackgroundBase,
                fHighlightColor,
                50
            );
        end;
        GetCanvas.Brush.Color := startc;
        GetCanvas.Pen.Color :=   dimColor(startc, 0.90);
        GetCanvas.Rectangle(r.Left,r.Top,r.Right,r.bottom);
    end;
    procedure LeftIcon;
    var ht, iconHT : integer;
        wd, iconWD : integer;
        bm : TBitmap;
        doicon : boolean;
        procedure DrawBox(checked : boolean);
        var
            center, chevronwd, chevronht : integer;
        begin
            copyrect(r,mainr);
            r.right := r.left+LEFT_SPACE;
            center := (r.Right+r.left) div 2;
            chevronwd := 14;
            chevronht := 14;
            inc(r.top, ((r.Bottom-r.Top) - chevronht) div 2);
            r.bottom := r.top + chevronht;
            r.left := (center - chevronwd div 2);
            r.Right := (center + chevronwd div 2);
            inc(r.right);

            GetCanvas.brush.Color := dimColor(fBackgroundColor,1.01);
            GetCanvas.Brush.Style := bsSolid;
            GetCanvas.pen.Color := saturate(clActiveCaption, 0.10);
            GetCanvas.Rectangle(r);

            GetCanvas.pen.Color := clBtnText;
            GetCanvas.Brush.Style := bsSolid;
            inc(r.Top, chevronht div 2);
            inc(r.Left, chevronwd div 2 - 3);
            if checked then begin
                GraphUtil.DrawCheck(GetCanvas,point(r.left,r.top),2);
            end;
        end;
        procedure DrawHIcon(wd,ht : integer; opacity:byte=255);
        var
            BlendFunc: TBlendFunction;
            clr : TColor;
        begin
            if (opacity=0) and not (p=hover) then EXIT;

            if (opacity=255) or (p=hover) then begin
                DrawIconEx(GetCanvas.handle,
                    r.left, r.top,
                    p.lefticon.icon, wd,ht, 0, 0, DI_NORMAL);
            end else begin
                if (p.LeftIcon.fIconBitmap = nil) then begin
                    p.LeftIcon.fIconBitmap := TBitmap.Create;
                    p.LeftIcon.fIconBitmap.Canvas.Lock;
                    p.LeftIcon.fIconBitmap.PixelFormat := pf24bit;
                    p.LeftIcon.fIconBitmap.Width := ICON_WIDTH;
                    p.LeftIcon.fIconBitmap.Height := ICON_HEIGHT;
                    p.LeftIcon.fIconBitmap.canvas.Brush.Color := fColumnColor;
                    if p.IsInExpandedMenu then begin
                        p.LeftIcon.fIconBitmap.canvas.Brush.Color := fExpandedBackgroundColor;
                    end;
                    p.LeftIcon.fIconBitmap.Canvas.FillRect(p.LeftIcon.fIconBitmap.Canvas.ClipRect);

                    DrawIconEx(p.LeftIcon.fIconBitmap.Canvas.Handle,
                        0, 0,
                        p.LeftIcon.icon, wd,ht, 0, 0, DI_NORMAL);

                    UnitMisc.desaturate(p.LeftIcon.fIconBitmap, DESATURATE_PERCENT, p.LeftIcon.fIconBitmap.canvas.Brush.Color);

                    BlendFunc.BlendOp := AC_SRC_OVER;
                    BlendFunc.BlendFlags := 0;
                    BlendFunc.SourceConstantAlpha := opacity;
//                    BlendFunc.SourceConstantAlpha := 255;
                    BlendFunc.AlphaFormat := 0;
                    Windows.AlphaBlend(GetCanvas.Handle, r.left,r.top, ICON_WIDTH,ICON_HEIGHT, p.LeftIcon.fIconBitmap.Canvas.Handle,
                        0,0, ICON_WIDTH,ICON_HEIGHT, BlendFunc);

                    p.LeftIcon.fIconBitmap.Canvas.CopyRect(rect(0,0,ICON_WIDTH,ICON_HEIGHT),
                    GetCanvas, rect(r.Left,r.Top,r.Left+ICON_WIDTH,r.Top+ICON_HEIGHT));
                    p.LeftIcon.fIconBitmap.Canvas.Unlock;
                end else begin
                    GetCanvas.Draw(r.Left,r.Top, p.LeftIcon.fIconBitmap);
                end;
            end;
        end;
        procedure DrawBitmap(x,y,wd,ht : integer; opacity:byte=255);
        var
            BlendFunc: TBlendFunction;
            clr : TColor;
        begin
            if (opacity=0) and not (p=hover) then EXIT;

            if (opacity=255) or (p=hover) and (not p.Disabled) then begin
                GetCanvas.StretchDraw(rect(x,y,x+wd,y+ht), p.LeftIcon.Bitmap);
            end else begin
                if (p.LeftIcon.fIconBitmap = nil) then begin
                    p.LeftIcon.fIconBitmap := TBitmap.Create;
                    p.LeftIcon.fIconBitmap.PixelFormat := pf24bit;
                    p.LeftIcon.fIconBitmap.Width := ICON_WIDTH;
                    p.LeftIcon.fIconBitmap.Height := ICON_HEIGHT;
                    p.LeftIcon.fIconBitmap.canvas.Brush.Color := fColumnColor;
                    if p.IsInExpandedMenu then begin
                        p.LeftIcon.fIconBitmap.canvas.Brush.Color := fExpandedBackgroundColor;
                    end;
                    p.LeftIcon.fIconBitmap.Canvas.FillRect(p.LeftIcon.fIconBitmap.Canvas.ClipRect);

                    p.LeftIcon.fIconBitmap.Canvas.Draw(0,0,p.LeftIcon.Bitmap);

                    UnitMisc.desaturate(p.LeftIcon.fIconBitmap, DESATURATE_PERCENT, p.LeftIcon.fIconBitmap.canvas.Brush.Color);

                    BlendFunc.BlendOp := AC_SRC_OVER;
                    BlendFunc.BlendFlags := 0;
                    BlendFunc.SourceConstantAlpha := opacity;
//                    BlendFunc.SourceConstantAlpha := 255;
                    BlendFunc.AlphaFormat := 0;
                    Windows.AlphaBlend(GetCanvas.Handle, r.left,r.top, ICON_WIDTH,ICON_HEIGHT, p.LeftIcon.fIconBitmap.Canvas.Handle,
                        0,0, ICON_WIDTH,ICON_HEIGHT, BlendFunc);

                    p.LeftIcon.fIconBitmap.Canvas.CopyRect(rect(0,0,ICON_WIDTH,ICON_HEIGHT),GetCanvas, rect(r.Left,r.Top,r.Left+ICON_WIDTH,r.Top+ICON_HEIGHT));
                end else begin
                    GetCanvas.Draw(r.Left,r.Top, p.LeftIcon.fIconBitmap);
                end;
            end;
        end;
        function getTop(ht, iconHT : Integer) : Integer;
        begin
            if (ht > iconHT + BLANK_SPACING)  then begin
                result := r.top + (ht - iconHT) div 2-1;
            end else begin
                Result := r.Top;
            end;
        end;
        function getLeft(wd, iconWD : Integer) : Integer;
        begin
            if (wd > iconWD + BLANK_SPACING)  then begin
                result := r.left + (wd - iconWD) div 2-1;
            end else begin
                result := r.left;
            end;
        end;
    begin
        copyrect(r,mainr);

        case p.Style of
        psNormal, psExpandable, psSubmenu : begin
            doicon := false;
            if not (p.LeftIcon.fIconOnlyOnHover or p.LeftIcon.fIconOnExpanded) then begin
                doicon := true;
            end;
            if p.LeftIcon.fIconOnlyOnHover and (p = Hover) then begin
                doicon := true;
            end;
            if p.LeftIcon.fIconOnExpanded and (p.Expanded = true) then begin
                doicon := true;
            end;
            if doicon then begin
                r := CalcIconRect(p, mainr);
                wd := r.width;
                ht := r.height;
                iconWD := ICON_WIDTH;
                iconHT := ICON_HEIGHT;
                bm := p.LeftIcon.Bitmap;
                if (hover=p) and (p.LeftIcon.HoverBitmap <> nil) and (self.FLastPosition = pipLeftIcon) then begin
                    bm := p.LeftIcon.HoverBitmap;
                end;
                if (bm<>nil) then begin
                    iconWD := bm.Width;
                    iconHT := bm.Height;
                end;
                if (iconHT > HT) then begin
                    iconWD := trunc((ht/iconHT) * iconWD);
                    iconHT := ht;
                end;

                CopyRect(iconr, r);
                r.Top := getTop(ht, iconHT);
                r.Left := getLeft(wd, iconWD);

                if (bm=nil) then begin
                   DrawHIcon(iconWD,iconHT, p.LeftIcon.fIconOpacity);
                end else begin
                    if (bm = p.LeftIcon.HoverBitmap) then begin
                        DrawBitmap(
                            r.Left,r.Top,
                            iconWD,iconHT,
                            p.LeftIcon.fIconOpacity
                        );
                    end else begin
                        DrawBitmap(
                            r.Left,r.Top,
                            iconWD,iconHT,
                            p.LeftIcon.fIconOpacity
                        );
                    end;
                end;
            end;
        end;
        psCheckable: begin
            if not p.IsCheckGrouped  then begin
                DrawBox(p.checked);
            end else if p.Checked then begin
                DrawBox(p.checked);
            end;
        end;
        end;
    end;
    procedure Caption;

    begin
        CopyRect(r,mainr);
        r.Left := CaptionLeft+1;

        case p.Style of
        psExpandable, psSubmenu:
            begin
                r.Right := r.Right - (RIGHT_SPACE+POPUP_RIGHT_BORDER);
            end;
        else
            r.right := r.Right - BLANK_SPACING;
        end;

        p.DrawCaption(self, GetCanvas, r, p=Hover, FLastPosition = pipLeftIcon);
    end;
    procedure CliptypeIcon;
    var center, cm : integer;
        chevronwd, chevronht, ht : integer;
        bm : TBitmap;
        chev : TRect;
        pc : TColor;
    begin
        copyrect(r,mainr);
        GetCanvas.brush.Style := bsSolid;
        case p.Style of
        psNormal:
            begin
                if (p.LeftIcon.CliptypeIcon <> nil) and
                not ((hover = p) and (Self.FLastPosition = pipLeftIcon))  then begin
                    ht := r.bottom-r.top;

                    GetCanvas.Brush.Color := fBackgroundColor;
                    r.Top := r.Top + ((ht-ICON_HEIGHT) div 2);

                    r.left := mainr.Left + getLeftSpaceWidth(p)-1;
                    r.Left := r.Left - p.LeftIcon.CliptypeIcon.Width;

                    GetCanvas.Draw(r.left, r.top, p.LeftIcon.CliptypeIcon);
                end;
            end;
        end;
    end;

    procedure RightIcon;
    var center : integer;
        chevronwd, chevronht, wd : integer;
        //bm : TBitmap;
        chev : TRect;
        pc : TColor;
    begin
        copyrect(r,mainr);
        r.Left := r.Right - (RIGHT_SPACE+POPUP_RIGHT_BORDER);

        GetCanvas.brush.Style := bsSolid;
        case p.Style of
        psSubmenu:
            begin
                if p.Disabled then begin
                    GetCanvas.pen.Color := fDisabledColor;
                end else begin
                    GetCanvas.pen.Color := fFontColor;
                end;
                GraphUtil.DrawArrow(GetCanvas,sdRight,
                    point(r.right-BLANK_SPACING*2-POPUP_RIGHT_BORDER-3,(r.top+r.bottom)div 2 -4),3);
            end;
        psExpandable:
            begin
                if p.Disabled then Exit;
                if p.Expanded  then begin
                    sd := sdUp;
//                    bm := ImgToBitmap(FrmMainPopup.ImgUp);
                end else begin
                    sd := sdDown;
//                    bm := ImgToBitmap(FrmMainPopup.imgDown);
                end;

                if not ((sd=sdup) and not p.ShowCollapseIcon) then begin

                    copyrect(r,mainr);
                    dec(r.right);
                    r.left := (r.right - RIGHT_SPACE) and not 1;
                    center := (r.Right+r.left) div 2;
                    chevronwd := 14;
                    chevronht := 14;
                    inc(r.top, ((r.Bottom-r.Top) - chevronht) div 2);
                    r.bottom := r.top + chevronht;
                    r.left := (center - chevronwd div 2)+1;
                    r.Right := (center + chevronwd div 2);
                    inc(r.right,2);

                    GetCanvas.brush.Color := clNone;
                    GetCanvas.Brush.Style := bsClear;

                    GetCanvas.pen.Color := blend(fbackgroundBase, fFontColor, 80);
                    GetCanvas.Rectangle(r);

                    CopyRect(chev, r);
                    inc(chev.Top, chevronht div 2 - 1);
                    inc(chev.Left, chevronwd div 2 - 3);
                    GetCanvas.Brush.Style := bsSolid;
                    if (p = hover) or (p.Expanded) then begin
                        pc := GetCanvas.Pen.Color;
                        GetCanvas.pen.Color := clBtnText;
                        GetCanvas.Brush.Color := clBtnFace;
                        // testing fillrect fix
                        //UseCanvas.FillRect(r);
                        MyFillRect(r);


                        
                    end else begin
                        if p.Disabled then begin
                            GetCanvas.pen.Color := clGrayText;
                        end;
                    end;
                    GraphUtil.DrawChevron(GetCanvas,sd,point(chev.left,chev.top),3);
                end;
            end;
        end;
    end;
    procedure DrawHoverLine;
    begin
    	copyrect(r,mainr);
        inc(r.Top);
        r.Left := CaptionLeft;
        dec(r.Right, BLANK_SPACING);
    	if (fDragStarted) and (p=fDropMenuItem)  then begin
            if IsLegalDrop(p, fDragMenuItem) then begin
               GetCanvas.pen.Color := fFontColor;
                    GetCanvas.MoveTo(r.Left,r.top);
                    GetCanvas.LineTo(r.right,r.top);
            end;
//            if (fDragMenuItem.ItemType = IT_POPUPCLIP) or (p.ItemType = IT_POPUPCLIP) then begin
//             	if (fDragMenuItem.ItemType = IT_POPUPCLIP) and (p.ItemType = IT_POPUPCLIP) then begin
//                    UseCanvas.pen.Color := fFontColor;
//                    UseCanvas.MoveTo(r.Left,r.top);
//                    UseCanvas.LineTo(r.right,r.top);
//                end;
//            end else if (fDragMenuItem.ItemType = IT_PERMANENT) or (p.ItemType = IT_PERMANENT) then begin
//             	if (fDragMenuItem.ItemType = IT_PERMANENT) and (p.ItemType = IT_PERMANENT) then begin
//                    UseCanvas.pen.Color := fFontColor;
//                    UseCanvas.MoveTo(r.Left,r.top);
//                    UseCanvas.LineTo(r.right,r.top);
//                end;
//            end else begin
//            	if not (p.Style = psLine) and not (p.ItemType in [IT_NONE])  then begin
//                    UseCanvas.pen.Color := fFontColor;
//                    UseCanvas.MoveTo(r.Left,r.top);
//                    UseCanvas.LineTo(r.right,r.top);
//                end;
//            end;
        end;
    end;
var
    i : integer;
    clr : TColor;
begin
    getCanvas.Lock;

    p.SetBoundsRect(self.Top, self.left);

    {main item rect}
    mainr.Top := p.fTop;
    mainr.Left := p.fLeft;
    mainr.Right := p.Right;
    mainr.Bottom := p.fbottom;

    if (not isTool(p)) then begin
        VerticalLines;
    end;
    inc(mainr.Left, POPUP_LEFT_BORDER);

    SeparatorLine;

    GetCanvas.Brush.Style := bsSolid;
    GetCanvas.Brush.Color := fBackgroundColor;

    i := GetCanvas.Font.Size;
    if p.SmallCaption then begin
        GetCanvas.Font.Size := i-1;
    end;




    LeftIconBackground;
    if (not isTool(p)) then begin
        LeftColumnEdge;
    end;
    CaptionBackground;

    if (p.IsBreak)  then begin
        {nothing needed}
    end else if (p.IsLine) then begin
        DrawLine;
    end else begin
        //HanldeDoubleHeightMarker;

        if (p = hover) and (not p.isGap) then begin
            DrawHoverHilite
        end;

        if getDrawLeftIcon(p) then begin
            LeftIcon;
            CliptypeIcon;
            PostDrawIconEvent(p, CalcIconRect(p, mainr));
        end;

        Caption;
        RightIcon;
    end;
    DrawHoverLine;
    HandleBottomBorder;

    if p = fLastVisible then begin
    	//DrawDragmark;
    end;

    GetCanvas.Font.Size := i;
    getCanvas.Unlock;
end;
procedure TACPopupPrototype.DrawCheckgroup(p: TACPopupItem);
var i : integer;
begin
    for i := 0 to p.fParent.fCheckgroup.count  - 1 do begin
        DrawItem( TACPopupItem(p.fParent.fCheckgroup[i]));
    end;
end;
procedure TACPopupPrototype.DrawDragmark;
var
	pa : Array of TPoint;
    r : TRect;
const wd = 7;
begin
    GetCanvas.Lock;
	with GetCanvas do begin
        brush.Color := clActiveCaption;
        pen.color := fFontColor;
        r := fmainrect;
        InflateRect(r,-1,-1);
        setlength(pa,3);
        pa[0] := point(r.Right-wd,r.bottom);
        pa[1] := point(r.Right, r.bottom-wd);
        pa[2] := point(r.Right, r.Bottom);
        polygon(pa);
    end;
    GetCanvas.Unlock;
end;
function TACPopupPrototype.getDrawLeftIcon(p : TACPopupItem) : Boolean;
begin
    result := true;
end;
procedure TACPopupPrototype.DrawOnCanvas(fullmode : boolean=false);
begin
	self.fFullMode := fullmode;

    GetCanvas.Font := fConfig.GetFont;
    GetCanvas.Font.Size := fConfig.GetFont.Size;
    GetCanvas.Font.Style := fConfig.getfont.style;

    self.CalcTotalSize;
    self.Paint;
end;
procedure TACPopupPrototype.PostDrawIconEvent(p : TACPopupItem; r : TRect);
begin
    // purposely blank
end;

procedure TACPopupPrototype.ReserveAccel;
var c : char;
begin
    c := self.GetAccelerator(0);
    while pos(c, fAccelCharsUsed) <> 0 do begin
        inc(fReservedCount);
    	c := self.GetAccelerator(0);
    end;
    inc(fReservedCount);
end;


procedure TACPopupPrototype.DeleteClipCallback(Sender: TObject);
begin
    self.RebuildPopup;
end;
procedure TACPopupPrototype.MakePermanentCallback(Sender: Tobject);
begin
    self.hide;
end;
procedure TACPopupPrototype.EditCallback(Sender: Tobject);
begin
    self.Hide;
end;
procedure TACPopupPrototype.PasteCallback(Sender: Tobject);
begin
    self.hide;
end;
procedure TACPopupPrototype.FormModeCallback(Sender : TObject);
begin
    if fPopupMode.equals(pdmStayOpen) then begin
        fPopupMode.setMode(pdmAutoHide);
    end else begin
        fPopupMode.setMode(pdmStayOpen);
    end;

    if (not self.UseFormMode) and fConfig.GetNeedsFocus then
            TFocusManager.ForceForeground(self.Handle);
end;


procedure TACPopupPrototype.DestroyCallback(Sender: Tobject);
begin
    self.RebuildPopup;
end;
procedure TACPopupPrototype.HideCallback(Sender : TObject);
begin
    self.ClipMenuHide;
end;
{//User Input//}

{Mouse Events}
procedure TACPopupPrototype.MouseCursorReset;
begin
    screen.Cursor := crDefault;
end;
procedure TACPopupPrototype.CMMouseLeave(var Message: TMessage);
begin
    HideTooltip;
    FLastPosition := pipNone;
end;
procedure TACPopupPrototype.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var p : TACPopupItem;

begin
    inherited;
    p := GetItemAt(x, y);

    if DragMouseUp(p) then EXIT;

    FrmDebug.AppendLog('MouseUp hover is nil='+BoolToStr(hover=nil));
    FrmDebug.AppendLog('MouseUp p is nil='+BoolToStr(p=nil));

    if p <> nil then  begin
        if (p = fMouseDownOn) then begin
            //if lastHitTest = HTCAPTION then EXIT;
            if p.Disabled  then EXIT;

            if (ssLeft in fDownShift) or (ssMiddle in fDownShift) then begin
                if (ssMiddle in fDownShift) then begin
                   p.MiddleClicked := true;
                end;
            end;

            self.fKeyActivated := false;
            HandleMouseClick(p, fDownShift);
        end;
    end else begin
        if (fPopupMode.equals(pdmPinned)) then begin
            if (GetForegroundWindow <> self.Handle) then begin
                TFocusManager.ForceForeground(self.Handle);
            end;
            self.RebuildPopup;
        end else if self.UseKeyboard then begin
            if (GetForegroundWindow <> self.Handle) then begin
                TFocusManager.ForceForeground(self.Handle);
                Windows.SetFocus(self.Handle);
            end;
        end;
    end;
    FrmDebug.AppendLog('MouseUp End');
end;
procedure TACPopupPrototype.MouseDown(Button: TMouseButton; Shift: TShiftState; X,Y: Integer);
begin
    inherited;

    CancelHovertimer;

    fDownShift := shift;
    fMouseDownOn := GetItemAt(x,y);



    self.DragMouseDown(Hover);
end;
procedure TACPopupPrototype.MouseLeave(Sender: TObject);
begin
    HideTooltip;
end;
procedure TACPopupPrototype.MouseMove(Shift: TShiftState; X, Y: Integer);
var last : TACPopupItem;
    lastpos : TPopupItemPosition;
    pt : TPoint;

    procedure ClearSubmenus(pc : TACPopupItem);
    begin
        repeat
             pc.fSubShowing := nil;
             pc := pc.fParent;
        until pc = nil;
    end;
    function IsChild(parent, pc : TACPopupItem) : boolean;
    begin

        result := false;
        if pc = nil then EXIT;

        repeat
             if Parent = pc.fparent then begin
                //result := true;
                 BREAK;
             end;

             pc := pc.fParent;
        until pc.fParent = nil;

        result := not (pc = fItems);
    end;
    function IsAutoScroll : boolean;
    var
        m : TMonitor;
    begin
        result := false;
        if (fIsTooTall) then begin
            pt := mouse.CursorPos;
            m := screen.MonitorFromPoint(pt);
            if m <> nil then begin
                pt.Y := pt.Y + m.top;
                if (pt.y = m.boundsrect.Top) or
                    (pt.y = m.BoundsRect.bottom+m.boundsrect.top-1) then begin
                    if (pt.y = m.boundsrect.Top) then begin
                        self.top := m.top;
                        fTopOffset := 0;
                    end else begin
                        fTopOffset := (self.Height - m.Height)*-1;
                    end;
                    self.Top := m.top + fTopOffset;
                    self.fTopOffsetEnable := true;
                    result := true;
                end;
            end;
        end;
    end;
var
    p : TACPopupItem;
begin
    inherited;

    if fIgnoreMouseMove then begin
        lastmousex := x;
        lastmousey := y;
    	EXIT;
    end;
    last := Hover;
    if IsAutoScroll then EXIT;

    if (Lastmousex=x) and (lastmousey=y) then begin
        {Mouse Move gets triggers after showing a tooltip, even though
        nothing has changed}
        EXIT;
    end;

    if self.fKeyActivated then begin
        self.fKeyActivated := false;
        self.ShowExpandedShortcuts := false;
        self.Invalidate;
    end;

    if not fDragStarted and (fDragMenuItem <> nil) then begin
        if (abs(fDragX - mouse.cursorpos.x) > mouse.DragThreshold) or
            (abs(fdragy - mouse.cursorpos.y) > mouse.DragThreshold) then begin
            fDragStarted := true;
            screen.Cursor := crDrag;
        end;
    end;

    LastMouseX := x;
    LastMouseY := y;

    p := GetItemAt(x,y);
    if (p <> nil) and p.IsLine then begin
        HideTooltip;
        //SetHover(p); // Windows will show no hover over a line
        if not p.IsInSubmenu  then begin
            timHideSubmenu.Enabled := true;
        end;
        if last <> nil then
            DrawItem(last);

        EXIT;
    end;
    if self.DragMouseMove(p, last) then EXIT;

    { perform hit test }
    lastpos := FLastPosition;
    FLastPosition := pipNone;
    if (p <> nil) then begin
        FLastPosition := pipCaption;
        if (x - p.fLeft) < getLeftSpaceWidth(p) then begin
            if (p.ClickableLeftIcon) then begin
                FLastPosition := pipLeftIcon;
            end;
        end;
    end;
    if (FLastPosition = pipNone) then begin
        timHideSubmenu.Enabled := true;
    end;
    if (lastPos <> FLastPosition) then begin
        if (hover<>nil) then begin
            {update to show or remove focus rectangle}
            DrawItem(hover);
        end;
        HideTooltip;
        StartHoverTooltip;
    end;
    if FLastPosition=pipLeftIcon then begin
        //ClipMenuShow
        if (hover <> nil) then begin
            timLeftIconHover.Enabled := false;
            timLeftIconHover.Enabled := true;
        end;
    end else begin
        timLeftIconHover.Enabled := false;
        self.ClipMenuHide;
    end;

    // update hover timers and set hover
    if (last <> p) then begin
        if (p <> nil) then begin
            self.CancelShowSubmenu;
            self.timMouseSelectDelay.Enabled := true;
            {current is a submenu, nested submenu}
            if (p.IsSubmenu) then begin
                if (p.IsInSubmenu)  then begin
                    {nested submenu}
                    {may not be possible}
                end else begin
                    {show me, if not showing}
                    if p <> p.fParent.fSubShowing then begin
                        self.ShowSubmenuDelayed(p);
                    end;
                end;
            end;
            if (not p.IsInSubmenu) then begin
                if (fLastSubmenu <> nil) and (fLastSubMenu <> p) then begin
                    timHideSubmenu.Enabled := true;
                end;
            end else begin
                if IsChild(p, fLastSubmenu) then begin
                    timHideSubmenu.Enabled := false;
                end else begin
                    // TODO: Figure out what the intent was here
                    {if (p <> fLastSubmenu) then begin
                        self.ShowSubmenuDelayed(p);
                    end;}
                end;
            end;

            SetHover(p);
        end;
    end;

    if not (FLastPosition = pipCaption) and (screen.Cursor > 0) then begin
        self.MouseCursorReset;
    end;
end;
procedure TACPopupPrototype.timMouseSelectDelayTimer(Sender: TObject);
begin
    timMouseSelectDelay.Enabled := false;
end;

{Keyboard Events}
procedure TACPopupPrototype.KeyDown(var Key: Word; Shift: TShiftState);

begin
    inherited KeyDown(key, shift);

    fLastKeyDown := key;
    fLastModiferDown := Shift;
    if (key in [VK_SHIFT, VK_MENU, VK_CONTROL]) and
        (hover<>nil) {and (hover.ItemType in TUsesClipMenu)} then begin
        if not (fToolTipNew.Showing and (ftooltipnew.lblHeader.Visible=false)) then begin
            timModifier.Enabled := true;
        end;
    end;


    if self.ClipMenuVisible then begin
        // consume the keystroke
        key := 0;

    end else begin
        // allow for auto-repeat using the Down message
        if (shift=[]) then begin
            fKeyActivated := true;

            case key  of
            VK_DOWN:    self.HoverDown;
            VK_UP:      self.HoverUp;
            VK_RIGHT:   self.HoverRight;
            VK_LEFT:    self.HoverLeft;
            else
                fKeyActivated := false;
            end;
        end;
    end;
end;
procedure TACPopupPrototype.KeyUp(var Key: Word; Shift: TShiftState);
var ch : char;
    i : integer;
    p : TACPopupItem;
begin
    inherited KeyUp(key, shift);

    if hover = nil then EXIT;

    case key of
    VK_MENU, VK_CONTROL, VK_SHIFT:
        begin
            // detect when only a modifier was pressed and released
            HideTooltip;
            CancelModifiertimer;

        end;
    VK_RETURN, VK_SPACE:
        begin
            self.fKeyActivated := true;
            self.HoverClick;
        end;
    end;

    if (fLastModiferDown <> []) and (fLastKeyDown <> 0) then begin
        i := fkeystrokes.IndexOf(string(char(fLastKeyDown)));
        if (i<>-1) then begin
            p := TACPopupItem(fkeystrokes.Objects[i]);
            p.fAltpressed := (ssAlt in fLastModiferDown);
            p.fShiftPressed := (ssShift in fLastModiferDown);
            p.fctrlpressed := (ssCtrl in fLastModiferDown);

            setHover(p);
            self.FLastPosition := pipCaption;
            self.HandleCaptionClick(p);
        end;
    end;
end;
procedure TACPopupPrototype.KeyPress(var Key: Char);
var i, j : integer;
    p, p2 : TACPopupItem;

    //it : TItemType;
begin
    inherited;
    if self.fIgnoreKeypresses then EXIT;

    if word(key) = VK_ESCAPE then begin
        self.ForceHide := true;
        self.Hide;
        EXIT;
    end;

    if self.ClipMenuVisible then begin
        Self.ClipMenuReportKey(key);
        key := #0;
        EXIT;
    end;


    if (self.fUseExpandedShortcuts and  self.fShowExpandedShortcuts) then begin
        i := fExpandedKeystrokes.IndexOf(string(key));
        if i <> -1 then begin
            self.fKeyActivated := True;
            p := TACPopupItem(fExpandedKeystrokes.Objects[i]);
            self.GatherModifierKeys(p);
            self.FLastPosition := pipCaption;
            if Assigned(p.OnClick) then
                while KeyboardQuery.IsPressed(VK_SHIFT) do begin
                    Application.ProcessMessages;
                    Sleep(50);
                end;

            PreHandleCaptionViaKeystroke(p, key);
            if p.IsSubmenu then begin
                SetHover(p);
                self.HandleCaptionClick(p);
            end else begin
                self.HandleCaptionClick(p);
            end;

            EXIT;
        end;
    end;



    i := fKeystrokes.IndexOf(string(key));
    if i <> -1 then begin
        self.fKeyActivated := true;
        p := TACPopupItem(fKeystrokes.Objects[i]);
        if p.Disabled then Exit;

        self.FLastPosition := pipCaption;
        PreHandleCaptionViaKeystroke(p, key);
        {highlight expandable items}
        if p.IsSubmenu then begin
            SetHover(p);
            self.CancelShowSubmenu;
            Self.HandleCaptionClick(p);
        end else if p.IsExpandable then begin
            if Assigned(p.OnClick) then
                while KeyboardQuery.IsPressed(VK_SHIFT) do begin
                    Application.ProcessMessages;
                    Sleep(50);
                end;

            self.HandleCaptionClick(p);
        end else begin
            self.HandleCaptionClick(p);
        end;
    end;

end;

{Click Event Processing}
procedure TACPopupPrototype.GatherModifierKeys(p : TACPopupItem);
begin
    p.fCtrlPressed := KeyboardQuery.IsPressed(VK_CONTROL);
    p.fShiftPressed := KeyboardQuery.IsPressed(VK_SHIFT);
    p.fAltpressed := KeyboardQuery.IsPressed(VK_MENU);
end;
function TACPopupPrototype.HandleMouseClick(p: TACPopupItem; DownShift : TShiftState) : boolean;
begin
    result := false;
	{handle Icon clicks, pass of Caption cliks}
    {handle right clicks}

    if ([ssLeft,ssMiddle] * fDownShift) <> [] then begin
        case FLastPosition of
            pipCaption: begin
                result := true;

                GatherModifierKeys(p);
                self.HandleCaptionClick(p);
            end;
        end;
    end;
end;
procedure TACPopupPrototype.HandleCaptionClick(p: TACPopupItem);
var wasChecked, wasExpanded, wasGrouped, wasExpandedShortcuts : boolean;
	i : integer;
    p2 : TACPopupItem;
    //it : TItemType;
    ci : TClipITem;
begin
    {handle special controls ourselves, pass off everything else to the click handler}
    case p.Style of
    psLine: ;
    psBreak: ;
    psSubmenu: begin
        p.SetShowingSubMenu;

        fExpandedKeystrokes.Clear;
        //fShowExpandedType := it;
        ShowExpandedShortcuts := self.fKeyActivated;
        self.AssignSubmenuShortcuts(p);

        self.RebuildPopup(false);
    end;
    psExpandable: begin
    	CancelHovertimer;
        //it := p.ItemType;
        //fAutoExpandOnce := IT_NONE;



        if p.fDisabled then EXIT;
        // detect expandables that handle their own event
        if not p.Expanded and assigned(p.CanExpand) then begin
            if not p.CanExpand(p) then EXIT;
        end;
        if p.Expanded and Assigned(p.CanCollapse) then begin
            if not p.CanCollapse(p) then Exit;
        end;


        FrmDebug.AppendLog('Normal Expand');

        wasExpanded := p.Expanded;
        //fShowExpandedType := it;


        wasExpandedShortcuts := fShowExpandedShortcuts;
        fShowExpandedShortcuts := false;
        if (fKeyactivated  and (fExpandedKeystrokes.Count = 0) and wasExpanded) then begin
            p.Expanded := false;
            fShowExpandedShortcuts := true;
            //fAutoExpandOnce := it;
            fKeyActivated := true;
            self.RebuildPopup(false);
            fKeyActivated := true;
        end else begin
            if (fConfig.getAutoCollapse and (not p.Expanded))
                 then begin
                for i := 0 to fitems.fSubMenu.count - 1 do begin
                    p2 := fitems.fSubMenu[i];
                    if (p2 = p) then CONTINUE;
                    if (p2.IsExpandable) and (p2.Expanded) then begin
                        if Assigned(p2.CanCollapse) then begin
                            if p2.CanCollapse(p2) then begin
                                FrmDebug.AppendLog('collapsing: '+p2.Caption);
                                p2.Expanded := false;
                            end;
                        end else begin
                            p2.Expanded := false;
                        end;
                    end;
                end;
                p.Expanded := true;
            end else begin
                p.Expanded := not p.Expanded;
            end;
            ShowExpandedShortcuts := self.fKeyActivated and p.Expanded;
        end;



        fExpandedKeystrokes.Clear;
        self.AssignSubmenuShortcuts(p);

        setHover(p);

        if Assigned(p.OnClick) then begin
            p.OnClick(p);
        end;
        RebuildPopup(false);


        //fAutoExpandOnce := IT_NONE;
    end;
    psCheckable: begin
    	wasChecked := p.Checked;
        p.Checked := not p.Checked;
        DrawItem(p);
        DrawCheckgroup(p);
        if assigned(p.OnClick) then begin
            p.OnClick(p);
        end;
        //todo: item may no longer exist
        //DrawItem(p);
    end;
    psNormal: begin
        case FLastPosition of
        pipLeftIcon:;
        pipNone: ;
        pipCaption: begin
            self.fCaptionPoint.X := self.left + p.Left;
            self.fCaptionPoint.Y := self.top + p.Top;

            if p.IsInExpandedMenu and p.fCollapseParentOnClick then begin
                if assigned(p.onclick) then begin
                    p.OnClick(p);
                    p.fParent.Expanded := false;
                    self.RebuildPopup(False);
                end;
            end else begin
                HandleOnClick(p);
            end;
        end;

        end;
    end;
    end;
end;
procedure TACPopupPrototype.HandleOnClick(p: TACPopupItem);
begin
    if assigned(p.OnClick) then begin
        if not p.StayOpenOnClick then
            self.Hide;
        p.OnClick(p);
    end;
end;
{Drag-and-Drop routines}
function TACPopupPrototype.IsLegalDrag(p : TACPopupItem) : boolean;
begin
    result := false;
end;
function TACPopupPrototype.IsLegalDrop(p, source: TACPopupItem): boolean;
begin
	result := false;

	if p = nil then EXIT;
    if fDragMenuItem = nil then EXIT;
	if p = fDragMenuItem then EXIT;

    result := true;
end;
procedure TACPopupPrototype.DragMouseDown(p: TACPopupItem);
begin
    if (p <> nil) and (fDragMenuItem = nil) then begin
        if IsLegalDrag(p) then begin
            fDragMenuItem := p;
            if p <> nil then begin
            	fDragX := mouse.cursorpos.x; fDragY := mouse.cursorpos.y;
            end;
        end;
    end;
end;
function TACPopupPrototype.DragMouseMove(p, last: TACPopupItem) : boolean;
var temp : TACPopupItem;
begin
	result := false;
    if (p <> nil) and fDragStarted then  begin
    	temp := fDropMenuItem;
        fDropMenuItem := p;

        if (p <> last) then begin
            DrawItem(p);
            if IsLegalDrop(p, fDragMenuItem) then begin


                screen.Cursor := crDrag;
            end else begin
                screen.Cursor := crNoDrop;
                FrmDebug.appendlog('no legaldrop');
            end;

            if temp <> nil then DrawItem(temp);
        end else begin
            if p = fDragMenuItem then begin
            	screen.Cursor := crNoDrop;
            end;
            if temp <> nil then DrawItem(temp);
            DrawItem(p);

        end;

        result := true;
    end else if fDragStarted  then begin
       	screen.Cursor := crNoDrop;
        temp := fDropMenuItem;
        fDropMenuItem := p;
		if temp <> nil then DrawItem(temp);
    end;
end;
function TACPopupPrototype.DragMouseUp(p: TACPopupItem) : boolean;

begin
	result := false;

    if fDragStarted then
        MouseCursorReset;

    if fDragStarted and (p<>nil) then begin
        fDragStarted := false;
        if IsLegalDrop(p, fDragMenuItem) then begin
            DragHandleDrop(fDragMenuItem, p);
        end else begin
        	DrawItem(p);
        end;
        fDragMenuItem := nil;
        fdragx := 0; fdragy := 0;

    	result := true;
    end else begin
    	fDragStarted := false;
        fDragMenuItem := nil;
        fDropMenuItem := nil;
    end;
end;
procedure TACPopupPrototype.DragHandleDrop(source, target : TACPopupItem);
begin
    // purposely blank, don't need an error so Abstract is not the correct usage
end;
{Timed Hide/Show Events}
procedure TACPopupPrototype.ShowSubmenuDelayed(p: TACPopupItem);
begin
    fQueuedSubmenu := p;
    timShowSubmenu.interval := 300;
    if timMouseSelectDelay.Enabled then begin
        timShowSubmenu.interval := 600;
    end;
    timShowSubmenu.Enabled := true;
end;
procedure TACPopupPrototype.timShowSubmenuTimer(Sender: TObject);
begin
    timShowSubmenu.Enabled  := false;

    if fQueuedSubmenu <> nil then begin
        fLastSubmenu := fQueuedSubmenu;
        fQueuedSubmenu.SetShowingSubMenu;
        if not fKeyActivated then
            fShowExpandedShortcuts := false;
        self.AssignSubmenuShortcuts(fQueuedSubmenu);
        fQueuedSubmenu := nil;



        self.RebuildPopup(false);
    end;
end;
procedure TACPopupPrototype.timHideSubmenuTimer(Sender: TObject);
    procedure ClearSubmenus(pc : TACPopupItem);
    begin
        if pc = nil then EXIT;

        repeat
             pc.fSubShowing := nil;
             pc := pc.fParent;
        until pc = nil;
    end;
var P : TACPopupItem;
begin
    timHideSubmenu.Enabled := false;
    timHideSubmenu.Interval := 300;

    p := fItems;
    if p.fSubShowing = nil then begin

    end else begin
        while p.fSubShowing <> nil do begin
            p := p.fSubShowing;
        end;
        fLastSubmenu := p;
        if fLastSubmenu <> nil then begin
            ClearSubmenus(fLastSubmenu);
            fLastSubmenu := nil;
            self.RebuildPopup(false);
        end;
    end;
end;
procedure TACPopupPrototype.timHoverTimer(Sender: TObject);
var pt : TPoint;

    r : TRect;
    s : string;
    hotkeyname : string;


begin
    CancelHovertimer;

    if hover = nil then EXIT;
    if FLastPosition = pipNone then EXIT;
	if fDragStarted then EXIT;



    if hover.IsExpandable and fConfig.getAutoExpandOnHover then begin
    	if not hover.expanded then begin
        	self.HoverClick;
        end;
        CancelHovertimer;
    	EXIT;
    end;


    ShowTooltip(hover);
end;
procedure TACPopupPrototype.timIgnoreMouseMoveTimer(Sender: TObject);
begin
	fIgnoreMouseMove := false;
    timIgnoreMouseMove.Enabled := false;
end;
procedure TACPopupPrototype.timLeftIconHoverTimer(Sender: TObject);
begin
    timLeftIconHover.Enabled := false;

    if FLastPosition = pipLeftIcon then
    	self.ClipMenuShow;
end;
procedure TACPopupPrototype.timModiferTimer(Sender: TObject);
var
    r : TRect;
    pt : TPoint;
    i : integer;
begin
    CancelModifiertimer;
    if (hover = nil) then EXIT;


    r := hover.BoundRect;
    pt := point(r.Right - trunc((r.Right-r.Left) / 3.5), r.bottom + 2);
    i := fToolTipNew.MinWidth;

    fToolTipNew.MinWidth := 40;

    if KeyboardQuery.IsPressed(VK_SHIFT) then begin
        fToolTipNew.HideHeader;
        fTooltipNew.ShowTooltip( fConfig.getShiftActionText,pt);
        CancelHovertimer;
        EXIT;
    end else if KeyboardQuery.IsPressed(VK_MENU) then begin
        fToolTipNew.HideHeader;
        fTooltipNew.ShowTooltip( fConfig.getAltActionText,pt);
        CancelHovertimer;
        EXIT;
    end else if KeyboardQuery.IsPressed(VK_CONTROL) then begin
        fToolTipNew.HideHeader;
        fTooltipNew.ShowTooltip( fConfig.getCtrlActionText,pt);
        CancelHovertimer;
        EXIT;
    end else begin
        fToolTipNew.HideHeader;
    end;

    fToolTipNew.MinWidth := i;
end;

procedure TACPopupPrototype.ShowTooltip(p : TACPopupItem);
var
    r : TRect;
    pt : TPoint;
begin
    r := p.BoundRect;
    pt := point(r.Right - trunc((r.Right-r.Left) / 3.5), r.bottom + 2);
    if p.Hint <> '' then begin
        fToolTipNew.HideHeader;
        fToolTipNew.SingleLineOnce := true;
        fToolTipNew.SmallFontOnce := true;
        fTooltipNew.ShowTooltip(p.hint,pt);
    end;
end;
procedure TACPopupPrototype.CancelShowSubmenu;
begin
    fQueuedSubmenu := nil;
    timShowSubmenu.Enabled := false;
end;
procedure TACPopupPrototype.HideTooltip;
begin
    CancelHovertimer;
    CancelModifiertimer;
    fToolTipNew.Hide;
end;
procedure TACPopupPrototype.StartHoverTooltip;
begin
    CancelModifiertimer;
    CancelHovertimer;
    timHover.Enabled := true;
end;
{Highlight routines}
procedure TACPopupPrototype.DetectAutoScroll(top,bottom : integer);
var
	pt : TPoint;
    m : TMonitor;
begin
	if fIsTooTall then begin
    	if top <> -1 then begin
           	pt.y := top;
            pt.x := 0;
            pt := self.ClientToScreen(pt);

            m := screen.MonitorFromWindow(self.handle);
            if (m <> nil) then begin
                if pt.y < m.top then begin
                	fIgnoreMouseMove := true;
                    timIgnoreMouseMove.Enabled := true;
                    self.Top := m.Top;
                    EXIT;
                end;
            end;
        end;

        if bottom <> -1 then begin
            pt.Y := bottom;
            pt.x := 0;
            pt := self.ClientToScreen(pt);


            m := screen.MonitorFromWindow(self.handle);
            if (m <> nil) then begin
                pt.Y := pt.y + m.top;
                if pt.y > m.BoundsRect.Bottom  then begin
                	fIgnoreMouseMove := true;
                    timIgnoreMouseMove.Enabled := true;
                    self.Top := m.top - (self.Height - m.Height);
                end;
            end;
        end;
    end;
end;

procedure TACPopupPrototype.ClearHover;
begin
    fHover := nil;
end;

procedure TACPopupPrototype.SetHover(value: TACPopupItem; tooltip:boolean=true; redraw:Boolean=true);
var last : TACPopupItem;
begin
    // this is the must be the ONLY method that sets fhover directly

	if fShowing = false then EXIT;
    {hotkey pasting will call this method, even if the PopupItem hides the popup}

    last := fHover;
    fHover := value;
    HideTooltip;
    if tooltip and
        (last <> fHover) and (fHover <> nil) and (last <> nil) then begin
        StartHoverTooltip;
    end;


    if FLastPosition <> pipLeftIcon then begin
        self.ClipMenuHide;
    end;

    if (value <> nil) then begin
        // Detect when Shortcut Keys need to shift up a level
        if (fKeyActivated) and
            (last<>nil) and (last.IsSubmenu) then begin
            if fHover.IsInExpandedMenu and not fhover.IsSubmenu and last.IsSubmenu then begin
                //fShowExpandedType := fHover.ItemType;
                fShowExpandedShortcuts := True;
                Self.AssignSubmenuShortcuts(fhover.fparent);
                self.Invalidate;
            end;
        end;

        if (redraw) then begin
            if last<>nil then DrawItem(Last);
            if fhover <> nil then DrawItem(fHover);
        end;

        // detect the mouse moving from a submenu to a child - don't
        // close the submenu when mouse selected
        if fLastSubmenu <> nil then begin
            if timMouseSelectDelay.Enabled then begin
                if fLastSubmenu.IsSubmenu then begin
                    if (fhover.VisibleParent = fLastSubmenu) then begin
                        self.timHideSubmenu.Enabled := false;
                    end else begin
                        if (last<>nil) and last.IsInSubmenu and not Hover.IsInSubmenu then
                            self.timHideSubmenuTimer(self);
                    end;
                end;
            end else begin
                // don't close the submenu when moving from parent to child (via keyboard)
                if not (fLastSubmenu.IsSubmenu and (fhover.VisibleParent = fLastSubmenu)) then begin
                    self.timHideSubmenuTimer(self);
                end;
            end;
        end;
        if fhover.IsSubmenu then begin
            if (last<>nil) and (last.IsSubmenu) then begin
                if timMouseSelectDelay.Enabled then begin
                    self.CancelShowSubmenu;
                    self.timHideSubmenu.Enabled := false;
                    self.timHideSubmenu.Enabled := true;
                end else begin
                    self.CancelShowSubmenu;
                    self.timHideSubmenuTimer(self);
                end;
            end;

            self.ShowSubmenuDelayed(fHover);
        end else if fhover.IsInSubmenu then begin

        end else begin
            if timMouseSelectDelay.Enabled then begin
                self.CancelShowSubmenu;
                self.timHideSubmenu.Enabled := false;
                self.timHideSubmenu.Enabled := true;
            end else begin
                self.CancelShowSubmenu;
                self.timHideSubmenuTimer(self);
            end;
        end;
    end;
end;
procedure TACPopupPrototype.HoverClick;
begin
    if Hover <> nil then begin
        FLastPosition := pipCaption;
        GatherModifierKeys(Hover);
        HandleCaptionClick(Hover);
    end;
end;
procedure TACPopupPrototype.HoverDown;
var i : integer;
    daddy, p : TACPopupItem;
begin
    if fItems.fSubMenu.Count = 0 then begin
        FrmDebug.AppendLog('HoverDown: Empty Popup');
        EXIT;
    end;

    // init or select next item, no matter what it is
    if Hover = nil then begin
        i := 0;
        p := fItems.fSubMenu[i];
        daddy := fItems;
    end else begin
        daddy := Hover.VisibleParent;

        i := Hover.Index;
        i := (i + 1) mod daddy.fSubMenu.Count;
        p := daddy[i];
    end;

    while ((p.Style in [psLine, psBreak, psGap]) or (not p.Visible)) do begin
        i := (i + 1) mod daddy.fSubMenu.Count;
        p := daddy.fSubMenu[i];
    end;


    FLastPosition := pipCaption;
    SetHover(p);
    self.DetectAutoScroll(p.top,p.bottom);
end;
procedure TACPopupPrototype.HoverLeft;
var p : TACPopupItem;
    i : integer;
    procedure CheckForExpandable(p : TACPopupItem);
    begin
        if (p.IsInExpandedMenu) then begin
            ShowExpandedShortcuts := self.fKeyActivated;
            //fShowExpandedType := p.ItemType;
        end;
    end;
begin

    if hover = nil then EXIT;
    {close submenu, if in submenu}
    if Hover.IsInSubmenu then begin
        ShowExpandedShortcuts := False;

        SetHover(Hover.fparent);
        self.CancelShowSubmenu;
        timHideSubmenuTimer(self);

        CheckForExpandable(Hover);
        if Hover.fParent.IsExpandable then
            self.AssignSubmenuShortcuts(hover.fparent);
        self.Invalidate;
    end else if Hover.IsSubmenu and (fitems.fSubShowing <> nil) and (hover.fSubMenu.Count <> 0)  then begin
        // close submenu
        ShowExpandedShortcuts := False;
        CheckForExpandable(Hover);

        SetHover(hover);
        self.CancelShowSubmenu;
        timHideSubmenuTimer(self);

        Self.AssignSubmenuShortcuts(hover.fparent);
        self.Invalidate;

    end else begin
        if hover.ColumnIndex = 0  then EXIT;

        p := getItemAt(hover.fLeft - 5, (hover.fTop+ hover.fBottom)div 2);
        if p = nil then begin
            p := hover;
            i := p.Index;
            while (p.ColumnIndex = hover.ColumnIndex) or p.IsBreak or not p.Visible do begin
            	dec(i);
                p := p.visibleparent[i];
            end;
        end;

        i := p.Index;
        while (p.isline or not p.Visible ) and (i > 0)do begin
            dec(i);
            p := p.VisibleParent[i];
        end;
        if p <> nil then begin
        	FLastPosition := pipCaption;
            SetHover(p);
        end;
    end;
end;
procedure TACPopupPrototype.HoverRight;
var i : integer;
    p : TACPopupItem;
begin
    if hover = nil then EXIT;

    if hover.IsSubmenu then begin
        if (Hover.fSubMenu.Count > 0) then begin
            hover.SetShowingSubMenu;
            fLastSubmenu := Hover;
            self.AssignSubmenuShortcuts(hover);
            self.RebuildPopup(false);
            i := 0;

            while (hover.fSubMenu[i].isGap) do begin
                inc(i);
            end;
            SetHover( hover.fSubMenu[i] );
        end;
    end else begin
        p := getItemAt(hover.right + 5, (hover.fTop+ hover.fBottom)div 2);
        if p = nil then begin
            if fitems.fColWidth.Count > 1 then begin
                p := fitems.fSubMenu[fitems.fsubmenu.count-1];
            end else begin
                EXIT;
            end;
        end;

        i := p.Index;
        while (p.isline or p.isGap or not p.visible) and (i > 0)do begin
            dec(i);
            p := p.VisibleParent[i];
        end;
        if p <> nil then begin
        	FLastPosition := pipCaption;
            SetHover(p);
        end;
    end;

end;
procedure TACPopupPrototype.HoverUp;
var i : integer;

    daddy, p : TACPopupItem;

begin

    // init or select next item, no matter what it is
    if Hover = nil then begin
        i := 0;
        daddy  := fItems;
        p := fItems.fSubMenu[daddy.fSubMenu.Count - 1];
    end else begin
        daddy := Hover.VisibleParent;
        i := Hover.index;
        dec(i);
        if i < 0 then i := daddy.fSubMenu.Count - 1;
        p := daddy[i];
    end;

    while ((p.Style in [psLine, psBreak, psGap]) or not p.Visible) do begin
        dec(i);
        if i < 0 then i := daddy.fSubMenu.Count - 1;
        p := daddy.fSubMenu[i];
    end;



    FLastPosition := pipCaption;
    SetHover(p);
    self.DetectAutoScroll(p.top,p.bottom);
end;

{Control Events}
var isPainting : boolean;
procedure TACPopupPrototype.paint;
begin
    if isPainting then Exit;
    GetCanvas.Lock;
    try
        isPainting := true;
        Frmdebug.StartTimer;
        inherited;
        frmDebug.AppendLog('paint: lock count = '+IntToSTr(GetCanvas.LockCount));
        self.FullRedraw;
        FrmDebug.EndTimerLog('paint time');
    finally
        isPainting := false;
        GetCanvas.Unlock;
    end;
end;
procedure TACPopupPrototype.resize;
begin
    //inherited;
    {not handling gets rid of flicker}
    FrmDebug.AppendLog('resize');
end;
procedure TACPopupPrototype.Hide;
begin
	if fShowing = false then begin
    	FrmDebug.AppendLog('acpopup.hide called twice');
        EXIT;
    end;
    fShowing := false;

    //inherited;
    FrmDebug.AppendLog('acpopup hide start');
    Windows.ShowWindow(self.Handle, SW_HIDE);

    CancelHovertimer;
    timShowSubmenu.Enabled := false;
    timHideSubmenu.Enabled := false;
    timLeftIconHover.Enabled := false;


    if assigned(fTooltipNew) then HideTooltip;
    self.ClipMenuHide;

    fFullMode := false;
    fPopupMode.reset;

    self.visible := false;
    fFirstPaint := false;
    self.MouseCursorReset;
    self.fTopOffsetEnable := false;

    ShowCaptionShortcuts := false;
    //self.ClearHover;
    //

    FrmDebug.AppendLog('acpopup hide end');
end;
procedure TACPopupPrototype.WMActivate(var msg: tMessage);
var
    pt : TPoint;
    p : TACPopupItem;
    h : THandle;
begin

    if Application.Terminated then Exit;

    {ignore all the inactive messages that appear until we've had a chance
    to paint and be visible to the user}
    case msg.WParam of
    WA_INACTIVE: begin
        if (lastActiveState <> WA_INACTIVE) then  begin
            InactiveEvent;
        end;
        lastActiveState := WA_INACTIVE;
    end;
    WA_ACTIVE, WA_CLICKACTIVE: begin
        if (lastActiveState <> WA_ACTIVE) and self.Visible then begin
            ActiveEvent(msg.WParam = WA_CLICKACTIVE);
        end;
        lastActiveState := WA_ACTIVE;
    end;
    end;
end;
procedure TACPopupPrototype.WMGetDlgCode(var Msg: TMessage);
begin
	{Tell Windows we want arrow keys}
    inherited;
    Msg.Result:= Msg.Result or DLGC_WANTARROWS;
end;
procedure TACPopupPrototype.WMNCHitTest(var Message: TWMNCHitTest);
var pt : tpoint;
    r : TRect;
    i : integer;
    p : TACPopupItem;
const THRESHOLD = 6;
begin
	inherited;

    //
    // detech mouse on draggable location
    //
    if not self.fDragStarted then begin
        pt := point(message.XPos, message.YPos);
        pt := self.ScreenToClient(pt);
        CopyRect(r,fMainRect);
        r.Left := r.Right - THRESHOLD;
        r.Top := r.Bottom - THRESHOLD;

        if PtInRect(r, pt) then begin
            message.Result := HTCAPTION;
            EXIT;
        end else begin
            if ShowCaption then begin
                if (pt.Y < CaptionHeight) then   begin
                    p := GetItemAt(pt.X, pt.y);
                    if  (p = nil) or (p.isGap) then begin
                        message.result := HTCAPTION;
                    end else begin
                        message.result := HTCLIENT;
                    end;
                end;
            end else begin
                message.result := HTCLIENT;
            end;
        end;
    end;
end;
procedure TACPopupPrototype.WMSetCursor(var M: TWMSetCursor);
begin
	if m.CursorWnd = self.Handle then begin
        if m.HitTest = HTCAPTION then begin
            screen.Cursor := crSizeAll;
            m.Result := 1;
        end else begin
            if screen.Cursor <= 0 then begin
                screen.Cursor := crArrow;
                m.result := 1;
            end;
        end;
    end;
end;
procedure TACPopupPrototype.WMMove(var Msg : TMessage);
begin
   fpoint.X := self.Left;
   fpoint.Y := self.Top;
end;
//
// handle cases where popup needs to be updated because of a focus change
procedure TACPopupPrototype.ActiveEvent(clicked: Boolean);
var
    pt : TPoint;
    p : TACPopupItem;
begin
    if fPopupMode.equals(pdmPinned) and (clicked) then begin
        pt := ScreenToClient(mouse.CursorPos);
        p := GetItemAt(pt.X,pt.y);
        if (p<>nil) then
            SetHover(p);
    end;
    if (clicked) and (ClipMenuVisible) and (FLastPosition = pipLeftIcon) then begin
        p := GetItemAt(pt.X,pt.y);
        if (p<>nil) then begin
            SetHover(p);
            HandleOnClick(p);
        end;
    end;
end;
procedure TACPopupPrototype.InactiveEvent;
begin
    if self.FirstPaint then begin
        self.hide;
        if self.showing and not self.ClipMenuVisible then begin
            self.RebuildPopup;
        end;
    end;
end;

//
//{ TACPopupItemCollection }
//
function TACPopupItemCollection.Add: TACPopupItem;
begin
    result := TACPopupItem.Create;
    result.Visible := true;

    self.add(result);
end;
function TACPopupItemCollection.GetItem(Index: Integer): TACPopupItem;
begin
    result := TACPopupItem(inherited GetItem(Index));
end;
procedure TACPopupItemCollection.SetItem(Index: Integer; const Value: TACPopupItem);
begin
    inherited SetItem(Index, Value);
end;


{
//
// TACIcon
//
}
constructor TACIcon.Create;
begin
    self.fBitmap := nil;
    self.fHasIcon := false;
    self.fIcon := 0;
    self.fCliptypeBitmap := nil;
    self.fHasCliptypeIcon := false;
    self.fIconOpacity := 255;
end;
destructor TACIcon.Destroy();
begin
    if fIconBitmap <> nil then MyFree(fIconBitmap);
    inherited;
end;

procedure TACIcon.SetBitmap(value: TBitmap);
begin
    self.fHasIcon := true;
    self.fBitmap := value;
end;
procedure TACIcon.SetHoverBitmap(value: TBitmap);
begin
    self.fHoverBitmap := value;
end;
procedure TACIcon.SetIcon(value: hicon);
begin
    self.fHasIcon := value <> 0;
    fIcon := value;
end;
procedure TACIcon.SetCliptypeIcon(value: TBitmap);
begin
    self.fCliptypeBitmap := value;
    self.fHasCliptypeIcon := true;
end;
procedure TACIcon.SetFrom(icon : HICON);
begin
    SetIcon(icon);
end;
procedure TACIcon.SetFrom(Bitmap : TBitmap);
begin
    SetBitmap(Bitmap);
end;



//
// TACPopupMode
//

constructor TACPopupMode.Create;
begin
    flist := TStack<TPopupDisplayMode>.create;
    setMode(pdmAutoHide);
end;
destructor TACPopupMode.Destroy;
begin
    flist.Clear;
    FreeAndNil(flist);
end;
procedure TACPopupMode.setMode(mode : TPopupDisplayMode);
begin
    PopValue;
    flist.Push(mode);
    if (flist.Count = 1) then begin
        fbaseMode := mode;
    end;
end;
function TACPopupMode.getMode : TPopupDisplayMode;
begin
    result := flist.Peek;
end;
function TACPopupMode.IsAutoClose : boolean;
begin
    result := getMode in [pdmAutoHide];
end;
function TACPopupMode.equals(mode : TPopupDisplayMode) : boolean;
begin
    result := getMode = mode;
end;
procedure TACPopupMode.PushValue;
begin
    flist.Push(pdmAutoHide);
end;
function TACPopupMode.PopValue : TPopupDisplayMode;
begin
    result := pdmAutoHide;
    if flist.Count > 0 then
        result := flist.Pop;
end;
procedure TACPopupMode.SetOverride(mode: TPopupDisplayMode);
begin
    PushValue;
    setmode(mode);
end;
procedure TACPopupMode.ClearOverride;
begin
    PopValue;
end;
function TACPopupMode.getModeNoOverride;
begin
    result := self.fbaseMode;
end;
procedure TACPopupMode.reset;
begin
    flist.Clear;
    setMode(pdmAutoHide);
end;


{
//
// TACPopupItem
//
}


constructor TACPopupItem.Create();
begin
    inherited;

    fSubMenu := TACPopupItemCollection.Create(false);
    fCheckgroup := TObjectList.Create(false);

    fLeftIcon := TACIcon.create();
    fLeftIcon.fIconOpacity := 255;

    fColWidth := TIntList.Create;
    fStyle := psNormal;


    fShowCollapseIcon := True;


    self.fPrefix := '';
    self.fCaption := '';
    self.fHoverOnlyCaption := '';
    self.fClip := nil;

    inc(itemCnt);
end;
destructor TACPopupItem.Destroy;
var i : integer;
    ac : TACPopupItem;
begin
    fColWidth.Clear;
    fColWidth.Free;
    fCheckgroup.Free;

    fSubmenu.Free;
    fLeftIcon.Free;

    inherited;

    dec(itemCnt);
end;

{Drawing Routines}
function TACPopupItem.showPrefix(ShowCaptionShortcuts : Boolean) : boolean;
var
    ch : Char;
    shortcutIndex : integer;
    isShortcut : boolean;
begin
    result := not (((IsInSubmenu or IsInExpandedMenu) and (ItemType <> IT_POPUPCLIP))
                or((ItemType = IT_TOOL) and not ShowCaptionShortcuts));

    if (fac.fShowExpandedShortcuts) then begin
        if Length(fprefix) < 2 then begin
            result := false;
            EXIT;
        end;

        ch := fprefix[2];
        shortcutIndex := fac.fExpandedKeystrokes.IndexOf(string(ch));
        isShortcut := shortcutIndex<>-1;
        if not result then begin
            // draw normally skipped prefix, only if it's referenced
            result :=
                (
                    isShortcut
                    and
                    (fac.fExpandedKeystrokes.IndexOfObject(self) = shortcutIndex)
                );
        end else begin
            // draw "always drawn" prefix, only if not overriden
            result := not isShortcut;
        end;
    end;

end;
function TACPopupItem.getPrefixWidth(C : TCanvas) : Integer;
var i : integer;
begin
    result := 0;
    if not showPrefix(fac.ShowCaptionShortcuts) then EXIT;
    i := C.TextWidth('W');
    if (fPrefix <> '') then begin
        result := i;
    end;
    if (result <> 0) then begin
        result := BLANK_SPACING + Result  + (BLANK_SPACING*PREFIX_TRAILING_SPACE_MULTIPLE);
    end;
end;
procedure TACPopupItem.getTextAreas(c : TCanvas; mainr : TRect; var prefix : TRect; var caption : TRect);
var prefixwd : integer;
begin
    prefixwd := getPrefixWidth(c);
    prefix.Top := mainr.Top;
    prefix.Left := mainr.left;
    prefix.Right := prefix.Left;
    if prefixwd > 0 then begin
        prefix.right := prefix.Right+ prefixwd - (BLANK_SPACING*PREFIX_TRAILING_SPACE_MULTIPLE);
    end;
    prefix.Left := prefix.Left + BLANK_SPACING;

    prefix.Bottom := mainr.Bottom;

    caption.Top := mainr.Top;
    caption.left := mainr.Left + prefixwd + BLANK_SPACING;
    if (ItemType = IT_PROGRAM_MENU) then begin
        caption.left := caption.left + ICON_WIDTH + BLANK_SPACING;
    end;
    caption.Right := mainr.Right;
    if fCaptionTrailingBitmap <> nil then begin
        caption.Right := caption.Right- fCaptionTrailingBitmap.Width;
    end;
    caption.Bottom := mainr.Bottom;
end;
function TACPopupItem.DesiredWidth(ac : TACPopupPrototype; c : TCanvas) : integer;
var s : string;
    i,j : integer;
    wd : integer;
begin
    self.fac := ac;
    result := getPrefixWidth(c);

    if fCaptionTrailingBitmap <> nil then
        inc(result, fCaptionTrailingBitmap.Width);

    if (ItemType = IT_PROGRAM_MENU) then begin
        result := result + ICON_WIDTH + BLANK_SPACING;
    end;
    if (fClip<>nil) and (fClip.GetFormatType = FT_PICTURE) then begin
        // dib AAAxYYYY
        try
            s := fClip.GetFormatName;
            UnitToken.TokenString(s,')');
            s := UnitToken.TokenString(s,'x');
            result := result +  StrToInt(s);
        except
        end;
    end else begin
        if SmallCaption then begin
//            i := c.Font.Size;
        end;
        wd := c.TextWidth(fCaption);
        if (wd <> 0) then begin
            inc(result, BLANK_SPACING+ wd + (BLANK_SPACING*RIGHT_SPACE_PADDING_MULTIPLE));
        end;
        if SmallCaption then begin
//            c.Font.Size := i;
        end;
    end;
end;

procedure TACPopupItem.DrawCaption(ac:TACPopupPrototype; c : TCanvas; r : TRect; IsHovered, IsIcon :  boolean);
    var
        ClipboardLeft, prefixwidth : integer;
        wc : TWideChar;
        hasPrefix : boolean;

    procedure TextColorSet;
    begin
        c.brush.Style := bsClear;
        c.pen.Color := 0;
        if fDisabled then begin
            c.pen.Color := ac.fDisabledColor;
        end else begin
            c.pen.Color := ac.fFontColor;
            if (fClip <> nil) and fClip.CData.Clicked then begin
                c.pen.Color := ac.fClickedColor;
            end;
        end;

        SetTextColor(c.Handle, ColorToRGB(c.pen.Color));
    end;
    procedure DrawPic;
    var
        st : TStream;
        pnt : Pointer;
        sz : Int64;
        tempr : TRect;
        bm : TBitmap;

    begin
        if not assigned(Clip.CData.thumb) then begin
            Clip.BuildMenuCaption;
        end;
        CopyRect(tempr, r);
        if hasPrefix then begin
            tempr.Left := tempr.Left + getPrefixWidth(c);
        end;
        if assigned(Clip.CData.thumb) then begin
            //
            // Detect already scaled thumbs
            // Otherwise, just draw the entire picture or a crop of
            // the picture
            //
            with Clip.CData do begin
                if (thumb.Width = r.Width) and (thumb.Height = r.Height) then begin
                    c.Draw(tempr.Left,tempr.Top, thumb);
                end else if ((thumb.Width + BLANK_SPACING) = r.Width) and
                    (thumb.Height = r.Height) then begin
                    c.Draw(tempr.Left,tempr.Top, thumb);
                end else begin
                    tempr.Width := min(thumb.Canvas.ClipRect.Width, r.Width);
                    tempr.Height := min(thumb.Canvas.ClipRect.Height, r.Height);
                    if (tempr.Height < r.Height) then begin
                        tempr.Top := tempr.Top + ((r.Height-tempr.Height) div 2);
                    end;

                    c.CopyMode := cmSrcCopy;
                    c.CopyRect(tempr, thumb.Canvas, rect(0,0,tempr.width, tempr.Height));
                end;
            end;
        end else begin
            try
                bm := TBitmap.Create;
                clip.DibToBitmap(BM);
                c.CopyMode := cmSrcCopy;
                c.CopyRect(tempr,bm.canvas,rect(0,0,bm.Width,r.Height));
                bm.Free;
            except
                FrmDebug.AppendLog('DrawPic: couldn''t draw bitmap',true);
            end;
        end;

        // Windows is doing something shady. Unless the BitmapInfo and the bit
        // buffer are continguous, the StretchDIBits call is failing
//            DrawDib(usecanvas, p.Clip.GetStream, r);
    end;
    procedure DrawClipboardMark(left : Integer);
    begin
        with FrmMainPopup.imgPasteMini.Picture do begin

            c.Draw(
                left - 4,
                Max(r.Top + (r.Height - Bitmap.height) div 2, 0),
                Bitmap
            );
            clipboardleft := left + Bitmap.Width-BLANK_SPACING;
        end;
    end;
    function HandlePrefix(r:TRect) : boolean;
    var
        i : integer;
        ch : Char;
        function IsSkippedPrefix : boolean;
        begin
            result := not showPrefix(AC.ShowCaptionShortcuts);
        end;
        procedure DrawPrefix;
        var
            c1 : TColor;
            bs : TBrushStyle;
            c2 : TColor;
            square : TRect;
            ht : integer;
            atop : integer;
            len : integer;
        begin
            bs := c.Brush.Style;
            c2 := c.Brush.Color;
            c1 := c.Pen.Color;

            c.Brush.Style := bsSolid;
            c.Pen.Style := psSolid;
            if (ac.UseKeyboard) then begin
                wc.Appendunicode(trim(fPrefix));
            end else begin
                wc.Appendunicode(trim(fPrefix[2]));
            end;
            len := wc.StrLength;


            prefixwidth := c.TextWidth('W');
            CopyRect(square, r);
            square.right := square.left + prefixwidth;
            inc(square.Right,2);
            dec(square.Left,1);
            ht := c.TextHeight('WZ');
            atop := max(0,(((square.Bottom-square.Top)-(ht)) div 2));
            inc(square.top, atop);
            square.bottom := square.Top + ht;
            if IsHovered then begin
                c.Brush.Color := DimColorOrReverse(ac.fbackgroundBase, 0.86);
            end else begin
                c.Brush.Color := DimColorOrReverse(ac.fbackgroundBase, 0.92);
            end;

            c.FrameRect(square);

            prefixwidth := square.right-square.left;


            c.Brush.Color := c2;

            if c1 <> clGrayText then begin
                c.Pen.Color := DimColorOrReverse(ac.fFontColor, 1.05);
            end else begin
                c.Pen.Color := c1;
            end;
            if (not ac.UseKeyboard) then begin
                c.Pen.Color := ac.fDisabledColor;
            end;

            SetTextColor(c.Handle, ColorToRGB(c.pen.Color));

            r.Right := r.Left + prefixwidth;
            c.Brush.Style := bsClear;
            square.Top := r.Top-1;
            square.Bottom := r.Bottom;
            DrawTextW(c.Handle, PWideChar(wc.Memory), len,
                square, DT_VCENTER or DT_SINGLELINE or DT_CENTER );

            SetTextColor(c.Handle, ColorToRGB(c.pen.Color));
            c.Pen.Color := c1;
            c.Brush.Style := bs;
        end;
    begin
        prefixwidth := 0;
        result := false;
        if (fPrefix <> '') then begin
            wc := TWideChar.Create;

            if showprefix(ac.ShowCaptionShortcuts) then begin
                DrawPrefix;
                Result := true;
            end;
            {
            if (ac.fShowExpandedShortcuts) then begin
                if IsSkippedPrefix then begin
                    // draw normally skipped prefix, only if it's referenced
                    ch := fprefix[2];
                    i := ac.fExpandedKeystrokes.IndexOf(string(ch));
                    if (i<>-1) and (ac.fExpandedKeystrokes.IndexOfObject(self) = i)
                         then begin
                        DrawPrefix;
                        Result := true;
                    end;
                end else begin
                    // draw "always drawn" prefix, only if not overriden
                    ch := fprefix[2];
                    i := ac.fExpandedKeystrokes.IndexOf(string(ch));
                    if (i=-1) then begin
                        DrawPrefix;
                        Result := True;
                    end;
                end;
            end else if not IsSkippedPrefix then begin
                DrawPrefix;
                result := true;
            end;
            }
            myfree(wc);
        end;
    end;
    procedure HandlePosssibleIcon(r : TRect);
    var ht : integer;
        y : integer;
        left : integer;
        icon : TIcon;
        bm : TBitmap;
        iconWD, iconHT : integer;
        BlendFunc: TBlendFunction;
        backgroundr, iconr : TRect;

    begin
        if fCaptionIconOnlyOnHover and (not IsHovered) then EXIT;

        if (fItemType = IT_PROGRAM_MENU) or (fCaptionTrailingBitmap <> nil) then begin
            ht := r.Height;
            if ht < ICON_HEIGHT then begin
                y := r.top;
            end else begin
                y := r.Top + max(0, ((ht)-ICON_HEIGHT)div 2);
            end;

            if (ItemType <> IT_PROGRAM_MENU) then begin
                c.Draw(r.left, y, fCaptionTrailingBitmap);
            end else begin
                inc(y);
                left := r.Left;
                Inc(left, BLANK_SPACING);
                iconHT := min(ICON_HEIGHT, r.Height);
                iconWD := ICON_WIDTH;
                if (iconHT < ICON_HEIGHT) then begin
                    iconWD := Trunc((iconHT/ICON_HEIGHT)*ICON_WIDTH);
                end;

                icon := TIcon.Create;
                icon.Handle := ShellApi.DuplicateIcon(0, fLeftIcon.Icon);
                bm := TBitmap.Create;
                bm.Canvas.Lock;
                bm.Width := ICON_WIDTH;
                bm.Height := ICON_HEIGHT;
                bm.PixelFormat := pf24bit;
                bm.TransparentColor := clWhite;
                backgroundr.Left := left;
                backgroundr.Top := y;
                backgroundr.Width := ICON_WIDTH;
                backgroundr.Height := ICON_HEIGHT;
                iconr.Top := 0;
                iconr.Left := 0;
                iconr.Width := ICON_WIDTH;
                iconr.Height := ICON_HEIGHT;

                // draw the icon normal and then blend it with the background

                bm.Canvas.CopyRect(iconr, c, backgroundr);
                DrawIconEx(c.Handle,
                    left, y,
                    fleftIcon.icon, iconWD,iconHT, 0, 0, DI_NORMAL);

                BlendFunc.BlendOp := AC_SRC_OVER;
                BlendFunc.BlendFlags := 0;
                BlendFunc.SourceConstantAlpha := 120;
                BlendFunc.AlphaFormat := 0;
                Windows.AlphaBlend(c.Handle, left,y, iconWD,iconHT, Bm.Canvas.Handle,0,0, bm.Width,bm.Height, BlendFunc);

                bm.Canvas.Unlock;
                bm.Free;
                icon.Free;
            end;
        end;
    end;
    procedure HandleDoubleHeightMarkers(r : TRect);
    var
        bs : TBrushStyle;
        c1, c2 : TColor;

    begin
        EXIT;

        bs := c.Brush.Style;
        c2 := c.Brush.Color;
        c1 := c.Pen.Color;

        c.Brush.Style := bsSolid;
        c.Pen.Style := psSolid;

        c.Pen.Color := AC.fFontColor;


        c.Brush.Style := bs;
        c.Brush.Color := c2;
        c.Pen.Color := c1;
    end;


    function HasCaption : boolean;
    begin
        result := fcaption <> '';
    end;

var
    options : integer;
    fs : TFontStyles;
    s : string;
    len, fontsz : integer;
    bm : TBitmap;
    tempr : TRect;
    ht : integer;

    prefixr, captionr, righticonr : TRect;
begin
    c.Lock;
    TextColorSet;
    fac := AC;
    getTextAreas(c, r, prefixr, captionr);
    hasPrefix := HandlePrefix(prefixr);

    if (isOnClipboard) then begin
        clipboardleft := captionr.left;
        if not (IsPictureClip) then begin
            DrawClipboardMark(ClipboardLeft);
            captionr.left := ClipboardLeft;
        end;
    end;

    wc := TWideChar.Create;
    if HasCaption or (fClip = nil) or (fClip.GetFormat <> CF_UNICODETEXT) then begin
        if HasCaption or (fClip = nil) then begin
            s := fcaption;
        end else begin
            if fClip.GetFormat = CF_DIB then begin
                s := fClip.GetFormatName;
            end else begin
                s := fclip.GetAsPlaintext;
            end;
        end;

        wc.AppendUnicode(s);
        wc.Replace(WideChar(' '),{NO BREAK SPACE}WideChar($00A0));
    end else begin
        wc.AppendUnicode(fClip.GetAsPlaintext);
    end;

    fs := c.Font.Style;
    fontsz := c.Font.Size;
    if (IsHovered) and (fIconHoverCaption <> '') and (IsIcon) then begin
        // icon hover override
        wc.Clear;
        wc.Append(fIconHoverCaption);
        c.Font.Style := c.Font.Style + [fsBold];

        if fItemType in [IT_PERMANENT] then begin
        //    fCaptionTrailingBitmap := ImgToBitmap(FrmMainPopup.imgNewWindow);
        end;
    end else begin
        if fItemType in [IT_FULL, IT_PERMANENT] then begin
            fCaptionTrailingBitmap := nil;
        end;
    end;
    if IsHovered and (fHoverOnlyCaption <> '') and not IsIcon then begin
        wc.Clear;
        wc.Append(fHoverOnlyCaption);
        c.Font.Style := c.Font.Style + [fsBold];
    end;

    if CaptionBold then begin
        c.Font.Style := c.Font.Style + [fsBold];
    end;

    len := min(100, wc.StrLength);
    SetBkMode(c.Handle, TRANSPARENT) ;


    options := 0;
    if (fClip<>nil) and not fUseAutoPrefix then begin
        options := options or DT_NOPREFIX;
    end;
    if SmallCaption then begin
        c.Font.Size := c.Font.Size - 1;
        options := options or DT_CENTER;
    end;

    // rect2 is either the size of the caption, or smaller if the picture is smaller

    TextColorSet;

    if (FrmConfig.cbNoThumbnails.Checked=false) and IsPictureClip then begin
        DrawPic;
        if IsOnClipboard then begin
            DrawClipboardMark(captionr.Left);
        end;
    end else begin
        if wc.size <> 0 then begin
            if DoubleHeight then begin
                // DT_VCENTER only works for single line text
                DrawTextW(c.Handle, PWideChar(wc.Memory), len,
                    captionr, DT_VCENTER or DT_WORDBREAK or DT_EDITCONTROL or options  );
                HandleDoubleHeightMarkers(r);
            end else begin
                inc(captionr.Bottom);
                    DrawTextW(c.Handle, PWideChar(wc.Memory), len,
                        captionr, DT_VCENTER or DT_SINGLELINE or options);
            end;
        end;
    end;

    CopyRect(righticonr, r);
    if (fItemType = IT_PROGRAM_MENU) then begin
        righticonr.Left := prefixr.right + BLANK_SPACING;
        righticonr.Right := r.Left + ICON_WIDTH + BLANK_SPACING;
    end else begin
        righticonr.Left := captionr.Right+1;
    end;

    HandlePosssibleIcon(righticonr);
    myfree(wc);

    SetBkMode(c.Handle, OPAQUE);
    c.Font.Style := fs;
    c.Font.Size := fontsz;

    c.Unlock;
end;
procedure TACPopupItem.DrawCaptionBackground(AC: TACPopupPrototype; c : TCanvas; r : TRect);
const
    BLEND_CLICKED = 97;
    BLEND_CLICKED_ORIGINAL = 99;
    DIVIDER_HALF_WIDTH = 60;
var
    c1,c2 : TColor;
    liner : TREct;
    m : integer;
    procedure MyFillRect(r : TRect);
    begin
        inc(r.bottom);
        inc(r.Right);
        c.FillRect(r);
    end;
    procedure MyGradientFill(c1, c2 : TColor; r : TRect; d : TGradientDirection);
    begin
        inc(r.Bottom);
        inc(r.Right);
        GraphUtil.GradientFillCanvas(c,c1,c2,r,d);
    end;
    procedure SetBrush(color : TColor);
    begin
        if DoubleHeight then begin
            c.Brush.Color := Blend(color, ac.fColumnColor, 80);
        end else begin
            c.Brush.color := color;
        end;
    end;
begin
    CopyRect(liner, r);
    if  (fItemType in [IT_CANCEL,IT_FULL]) then begin
        SetBrush(ac.fColumnColor);
        ac.fBackgroundBase := c.brush.Color;
        MyFillRect(r);

        c1 := ac.fColumnColor;
        c2 := dimColor(ac.fColumnColor, 0.94);
        liner.Left := liner.right - RIGHT_MENU_GRADIENT_WIDTH;
    end else if IsInExpandedMenu then begin

        if (fClip <> nil) and fClip.CData.Clicked then begin
            c.Brush.Color := unitMisc.Blend(c.brush.Color, ac.fClickedColor, BLEND_CLICKED);
        end else begin
            SetBrush(ac.fExpandedBackgroundColor);
        end;
        ac.fBackgroundBase := c.brush.Color;
        MyFillRect(r);
    end else begin
        if (fItemType = IT_PINNED) then begin
            SetBrush(DimColorOrReverse(ac.fBackgroundColor,1.03));
            ac.fBackgroundBase := c.brush.Color;
        end else if (fItemType = IT_TOOL) then begin
            SetBrush(ac.fCaptionColor);
            ac.fBackgroundBase := c.brush.Color;
        end else begin
            SetBrush(ac.fBackgroundBase);
        end;
        MyFillRect(r);
    end;
end;

{utils}
function TACPopupItem.IsOnClipboard : boolean;
begin
    result :=  (fItemType = IT_CLIPBOARD);
    result := result or (
        (fItemType = IT_POPUPCLIP) and
        (fClip.CData <> nil) and
        (fClip.CData.Hash = CurrentClipboard.CData.Hash)
    );
end;
function TACPopupItem.IsPictureClip : boolean;
begin
    result :=
        //(fItemType in TContainsAClip) and
        (fClip<>nil) and
        (fClip.GetFormatType = FT_PICTURE);
end;
function TACPopupItem.getClickableLeftIcon : boolean;
begin
    result := fClickableLeftIcon and not Self.Disabled and fLeftIcon.fHasIcon;
end;



function TACPopupItem.GetBoundsRect: TRect;
begin
    result := fBoundsRect;
end;

function TACPopupItem.GetChecked: boolean;
begin
    result := fChecked;
end;
function TACPopupItem.GetExpanded: boolean;
begin
    result := fExpanded;
end;

function TACPopupItem.GetIndex: integer;
begin
    {The index refers to the visible parent, not the owner}
    result := GetVisibleParent.fSubMenu.IndexOf(self);
end;

function TACPopupItem.GetOwnerIndex : integer;
begin
    result := fParent.fSubMenu.IndexOf(self);
end;

function TACPopupItem.GetIsBreak: boolean;
begin
    result := fStyle = psBreak;
end;
function TACPopupItem.GetIsExpandable: boolean;
begin
    result := fStyle = psExpandable;
end;
function TACPopupItem.GetIsInExpandedMenu: boolean;
begin
    result := fParent.IsExpandable;
end;
function TACPopupItem.GetIsInSubMenu: boolean;
begin
    if fParent = nil then begin
        FrmDebug.AppendLog('narent fpull');
    end;
    result := (fparent.fParent <> nil) and not fparent.IsExpandable;
end;
function TACPopupItem.GetIsLine: boolean;
begin
    result := fStyle = psLine;
end;
function TACPopupItem.GetIsSubmenu: boolean;
begin
    result := fStyle = psSubmenu;
end;
function TACPopupItem.GetIsGap : boolean;
begin
    result := fStyle = psGap;
end;

function TACPopupItem.GetItem(Index : integer): TACPopupItem;
begin
    result := fSubMenu[index];
end;

function TACPopupItem.GetVisibleParent: TACPopupItem;
begin
    {In an expanded menu, the visible parent will be twice removed}
    result := fParent;
    if self.IsInExpandedMenu  then
        result := fParent.fParent;
end;
function TACPopupItem.getVisible : boolean;
begin
    result := fvisible;
    if IsInExpandedMenu and not fParent.Expanded then begin
        result := false;
    end;
end;

procedure TACPopupItem.CheckGrouped;
begin
    fParent.fCheckgroup.Add(self);
end;
function TACPopupItem.IsCheckGrouped : boolean;
begin
    result := fparent.fCheckgroup.IndexOf(self) <> -1;
end;
function TACPopupItem.Add: TACPopupItem;
begin
    result := self.fSubMenu.Add;
    result.fParent := self;

    if self.IsSubmenu then begin
        self.Disabled := false;
    end;

    if IsExpandable then begin
        fParent.fSubMenu.Add(result);
        self.fSubMenu.OwnsObjects := false;
    end;
end;


function TACPopupItem.AddBreak: TACPopupITem;
begin
    result := self.add;
    result.IsBreak := true;
end;
function TACPopupItem.CalcRight: integer;
begin
    result := self.fRight;
    EXIT;

    // legacy stuff
    if self.fStaticRight then begin
        result := self.fRight;
        EXIT;
    end;
    if self.fParent.IsExpandable  then begin
        result := self.fLeft + self.fParent.fparent.fColWidth[self.fMyColumn] - POPUP_RIGHT_BORDER -1;
    end else begin
        result := self.fLeft + self.fParent.fColWidth[self.fMyColumn] - POPUP_RIGHT_BORDER - 1;
    end;
    self.fRight := result;
end;
procedure TACPopupItem.SetBoundsRect(parenttop, parentleft: integer);
begin
    with fBoundsRect do begin
        top := ftop + parenttop;
        bottom := fbottom + parenttop;
        left := fleft + parentleft;
        right := self.right + parentleft;
    end;
end;

procedure TACPopupItem.SetCaption(value: string);
begin
    fCaption := value;
    if (length(fCaption) > 300) then begin
        fCaption := leftstr(fCaption, 300) + '...';
    end;
    // tabs make the windows too damn long
    // the #10 will make a funky box in the tab's place
    while pos(#9#9, fCaption) > 0 do begin
        fCaption := stringreplace(fCaption, #9#9, #9, [rfreplaceall]);
    end;
    fCaption := stringreplace(fCaption, #9,chr($B0), [rfReplaceAll]); {unicode middle dot}

    {escape the special hotkey character}
    if not fUseAutoPrefix then begin
        fCaption := stringreplace(fCaption, '&', '&&', [rfReplaceAll]);
    end;
end;
procedure TACPopupItem.SetCaptionFromClip(value: TClipItem);
begin
    if value.CData.displaytext <> '' then begin
        fCaption := value.CData.displaytext;
        EXIT;
    end;
    case value.GetFormatType of
    FT_UNICODE, FT_RICHTEXT, FT_HTML:
        begin
            value.BuildMenuCaption;
            fCaption := value.CData.displaytext;
        end;
    FT_PICTURE:
        begin
            fCaption := value.GetFormatName()
        end;
    else
        begin
            SetCaption(value.GetAsPlaintext);
        end;
    end;
    value.CData.displaytext := fCaption;
end;

procedure TACPopupItem.SetChecked(value: boolean);
var i : integer;
    p : TACPopupItem;
begin
    if (fParent.fCheckgroup.IndexOf(self) <> -1) then begin
        if (self.fChecked = false) and value then begin
            for i := 0 to fParent.fCheckgroup.Count - 1 do begin
                p := TACPopupItem(fParent.fCheckgroup[i]);
                p.fChecked := false;
            end;
            self.fChecked := true;
        end;
    end else begin
        self.fChecked := value;
    end;

    self.fStyle := psCheckable;
    self.fStayOpenOnClick := true;
end;
procedure TACPopupItem.SetExpanded(value: boolean);
var i : integer;
begin
    if FExpanded = value then EXIT;

    fExpanded := value;

//    if value then begin
//        for i := fSubMenu.count - 1 downto 0 do begin
//            fparent.fSubMenu.Insert(self.index+1,fsubmenu[i]);
//        end;
//    end else begin
//        for i := fSubMenu.count - 1 downto 0 do begin
//            fparent.fSubMenu.Extract(fSubmenu[i]); // Don't Free the object
//        end;
//    end;
end;

procedure TACPopupItem.SetIsBreak(value: boolean);
begin
    if value then
        fStyle := psBreak;
end;
procedure TACPopupItem.SetIsExpandable(value: boolean);
begin
    if value then begin
        fStyle := psExpandable;
        fLeftIcon.fIconOnlyOnHover := false;
        fLeftIcon.fIconOnExpanded := true;
        fStayOpenOnClick := true;
    end;
end;
procedure TACPopupItem.SetIsLine(value: boolean);
begin
    if value then begin
        fStyle := psLine;
        fStayOpenOnClick := true;
    end;
end;
procedure TACPopupItem.SetIsSubmenu(value: boolean);
begin
    if value then begin
        fStyle := psSubmenu;
        fStayOpenOnClick := true;
    end;
end;
procedure TACPopupItem.SetIsGap(value : boolean);
begin
    if value then begin
        fStyle := psGap;
        fStayOpenOnClick := true;
    end;

end;
procedure TACPopupItem.SetItem(Index: integer; value: TACPopupItem);
begin
    fSubMenu[index] := value;
end;


procedure TACPopupItem.SetShowingSubMenu;
begin
    if self.IsInExpandedMenu  then begin
        fParent.fParent.fSubShowing := self;
    end else begin
        fParent.fSubShowing := self;
    end;

end;


procedure TACPopupItem.SetHoverOnlyCaption(value : string);
begin
    self.fHoverOnlyCaption := value;
end;
function TACPopupItem.GetHoverOnlyCaption : string;
begin
    result := fHoverOnlyCaption;
end;


//
// Handling transparency
//
constructor TCustomTransparentControl.Create(AOwner: TComponent);
begin
    inherited Create(AOwner);
    ControlStyle := ControlStyle + [csOpaque];
    Brush.Style := bsClear; {required for XP to make backgroun transparent}
end;
procedure TCustomTransparentControl.CreateParams(var Params: TCreateParams);
begin
    inherited CreateParams(Params);

    // dropshadow
    if CheckWin32Version(5, 1) then
       params.WindowClass.Style := params.WindowClass.style or CS_DROPSHADOW;
end;
procedure TCustomTransparentControl.WMNCHitTest(var Message: TWMNCHitTest);
begin
    if not FInterceptMouse then begin
        Message.Result := HTTRANSPARENT
    end else begin
        inherited;
    end;
end;
procedure TCustomTransparentControl.Invalidate;
begin
    FrmDebug.AppendLog('invalidate');
	inherited Invalidate;
end;

initialization
finalization
begin
    TFrmDebug.MeOnlyAppend('UnitCACPopupPrototype', false);
end;


end.
