unit UnitAutoBackup;


interface

uses System.zip, Generics.Collections, System.Classes;

type TAutoBackup = class(TObject)
    private
        fZip : TZipFile;
        fClosed : boolean;
        fErrors : TList<String>;
        function GetNewStream(filename : string) : TFileStream;
        class procedure BackupCanClose(Sender: TObject; var CanClose: Boolean);
    public
        constructor Create(BackupFullName : string);
        destructor Destroy; override;
        procedure Close;

        class function GetBackupName(programInitials : string) : string;
        function AddToBackup(DataPath : string) : boolean;
        property Errors : TList<String> read fErrors;

        class procedure DialogBackup(WinHandle : THandle; ProgramInitials : string; DialogPath : string; backupPaths : TList<String>);
end;


implementation

uses WIndows, messages, Forms, SysUtils,  System.IOUtils, Dialogs;

var major, minor : integer;
var build : string;

constructor TAutoBackup.Create(BackupFullName : string);
begin
    fzip := TZipFile.Create;
    fzip.Open(BackupFullName, zmWrite);
    ferrors := TList<String>.create;
end;
destructor TAutoBackup.Destroy;
begin
    FreeAndNil(ferrors);
    FreeAndNil(fzip);
    FreeAndNil(ferrors);
end;
procedure TAutoBackup.Close;
begin
    fzip.Close;
    ferrors.Clear;
end;

class procedure TAutoBackup.DialogBackup(WinHandle : THandle; ProgramInitials : string; DialogPath : string; backupPaths : TList<String>);
var s : string;
    ab : TAutoBackup;
    backupFullName : string;
    sdBackup : TSaveDialog;
    i : integer;
begin
    sdbackup := TSaveDialog.Create(nil);

    sdBackup.Options := sdBackup.Options + [ofPathMustExist];
    sdBackup.Filter := 'Zip (*.zip)|*.zip';
    sdBackup.DefaultExt := 'zip';

    sdBackup.InitialDir := DialogPath;
    sdBackup.FileName := TAutoBackup.GetBackupName(ProgramInitials);
    sdBackup.OnCanClose := TAutoBackup.BackupCanClose;
    if sdBackup.Execute(WinHandle) then begin
        if FileExists(sdBackup.Files[0]) then begin
            ShowMessage('File already exists. Backup not created.');
        end else begin
            backupFullName := sdBackup.Files[0];
            ab := TAutoBackup.Create(backupFullName);

            for s in backupPaths do begin
                ab.AddToBackup( s );
            end;
            ab.Close;
            FreeAndNil(ab);
            ShowMessage('Backup Complete'+#13#10+backupFullName);
        end;
    end;

    FreeAndNil(sdBackup)
end;
class procedure TAutoBackup.BackupCanClose(Sender: TObject; var CanClose: Boolean);
var sd : TSaveDialog;
begin
    sd := TSaveDialog(Sender);


    CanClose := (not FileExists(sd.Files[0]));

    if not CanClose then begin
        ShowMessage('File already exists.');
    end;
end;
class function TAutoBackup.GetBackupName(programInitials : string) : string;
begin
    result := programInitials + 'v'+IntToStr(major);
    if minor < 10 then begin
        result := result + '0';
    end;
    result := result + IntToStr(minor) + ' '+build+'.zip';
end;

function TAutoBackup.AddToBackup(DataPath : string) : boolean;
var startingFolder : string;
    procedure ScanAndZip(path : string);
    var s, filename, subfolder : string;
        fa : TFileAttributes;
        err : boolean;
        fs : TFileStream;
    begin
        for s in TDirectory.GetFileSystemEntries(path) do begin
            filename := TPath.GetFileName(s);
            fa := TPath.GetAttributes(s);
            if TFileAttribute.faDirectory in fa then begin
                if not ((filename = '.') or (filename = '..')) then begin
                    ScanAndZip( IncludeTrailingPathDelimiter(s) );
                end;
            end else begin
                if TPath.GetExtension(s) = '.zip' then CONTINUE;
                try
                    subfolder := TPath.GetDirectoryName(s);
                    Delete(subfolder, 1, length(startingFolder));
                    err := false;
                    if subfolder <> '' then begin
                        fZip.add(s, IncludeTrailingPathDelimiter(subfolder) + filename);
                    end else begin
                        fZip.Add(s);
                    end;
                except on e: Exception do
                    begin
                        err := true;
                        try
                            fs := Self.GetNewStream(s);
                            if subfolder <> '' then begin
                                fZip.add(fs, IncludeTrailingPathDelimiter(subfolder) + filename);
                            end else begin
                                fZip.Add(fs, filename);
                            end;
                            FreeAndNil(fs);
                        except
                            ferrors.Add(E.message);
                        end;
                    end;
                end;
            end;

            if (err) then begin

            end;
        end;
    end;
begin
    startingFolder := IncludeTrailingPathDelimiter(DataPath);
    ScanAndZip(startingFolder);
end;
function TAutoBackup.GetNewStream(filename : string) : TFileStream;
begin
    result := TFileStream.Create(filename, fmShareDenyNone or fmOpenRead);
end;

function getAppFolder : string;
begin
    result := IncludeTrailingPathDelimiter(
        TPath.GetDirectoryName(Application.ExeName)
    );
end;
procedure GetAppVersion(var major, minor : integer; var build : string);
var L, i : DWORD;
    p, buf : Pointer;
    ver : array [0 .. 3] of word;
begin
    L := Windows.GetFileVersionInfoSize(PChar(Application.EXEName), i);
    if (L = 0) then begin
        EXIT;
    end;


    GetMem(buf, L);
        Windows.GetFileVersionInfo(PChar(Application.EXEName), 0, L, Buf);
        Windows.VerQueryValue(Buf, '\', p, i);

        Ver[0] := HiWord(TVSFixedFileInfo(p^).dwFileVersionMS);
        Ver[1] := LoWord(TVSFixedFileInfo(p^).dwFileVersionMS);
        Ver[2] := HiWord(TVSFixedFileInfo(p^).dwFileVersionLS);
        Ver[3] := LoWord(TVSFixedFileInfo(p^).dwFileVersionLS);

    FreeMem(Buf);

    Major := ver[0];
    Minor := ver[1];
    Build := IntToStr(ver[2]) +'-' + IntToStr(ver[3]);
end;

initialization
begin
    GetAppVersion(major, minor, build);
end;


end.
