unit UnitClipboardGrabber;

interface
{
    Handles all the details of collecting information from the Clipboard.

    Also automatically retries for a period of time if the clipboard is locked
    by another program. (eg. Excel does this)
}

uses UnitClipQueue, Windows, Classes, System.SyncObjs;

type TClipboardGrabber = class(TObject)
private

    class var fLastDelay : integer;
    class var fLock : TMutex;
    class procedure InitLock;
    class procedure CreateLock;
    class procedure ReleaseLock;

    class var fClipboardOpen : boolean;
    class function TryOpenClipboard : boolean;
    class procedure CloseClipboard;

    class function ChooseAFormat : TClipFormat;
    class procedure LastDitchAttempt(ci : TClipItem);
    class function CleanRichtextURLs(s : string) : string;
    class function GetAvailableFormats : TClipFormatList;
    class function GetConfiguredTextFormat(listOrNil : TClipFormatList) : TClipFormat;
public
    class function GetClipboardItem(clip : TClipItem; hi : HICON; OverrideFormat : WORD = 0; SizeLimit : cardinal = $FFFF) : cardinal; overload;
    class function ClipboardHasPicAndText(var textformat :word) : boolean;
    class function asText : string;

    class function getLastDelay : integer;
    class function hasFormat(format : Word) : boolean;
end;


implementation

uses UnitFrmDebug, UnitFrmConfig,
SysUtils, UnitMisc, Clipbrd, Forms, UnitToken,
Generics.Collections, UnitMyClipboard;


type TClipItemHelper = class helper for TClipItem
    private
        function getRichShadowMemory : TStream;
        procedure setRichShadowMemory(value : TStream);
    public
        function GetStreamRaw_ : TStream;
        function GetFilenamesAsText_(h : THandle) : string;
        function GetCRC_ : cardinal;
        procedure setFormat(value : Word);
        //property cformat : Word read getFormat write setFormat;
        property RichShadowMemory_ : TStream read getRichShadowMemory write setRichShadowMemory;
end;

type TClipDataHelper = class helper for TClipData
    private
        function getPictureSize : string;
        procedure setPictureSize(value : string);
        function gethndIcon : THandle;
        procedure sethndIcon(value : THandle);
        function gettimestamp : TDateTime;
        procedure settimestamp(value : TDateTime);
        function gettimestampused : Boolean;
        procedure settimestampused(value : Boolean);
    public
        property hndIcon_ : THandle read gethndIcon write sethndIcon;
        property picturesize_ : string read getPictureSize write setPictureSize;

        property timestamp_ : TDateTime read gettimestamp write settimestamp;
        property timestampused_ : Boolean read gettimestampused write settimestampused;
end;

procedure TClipItemHelper.setFormat(value : Word);
begin
    Self.cformat := value;
end;
function TClipItemHelper.getRichShadowMemory : TStream;
begin
    Result := Self.RichShadowMemory;
end;
procedure TClipItemHelper.setRichShadowMemory(value : TStream);
begin
    Self.RichShadowMemory := value;
end;
function  TClipItemHelper.GetStreamRaw_ : TStream;
begin
    result := self.GetStreamRaw;
end;
function TClipItemHelper.GetFilenamesAsText_(h : THandle) : string;
begin
    result := self.GetFilenamesAsText(h);
end;
function TClipItemHelper.GetCRC_ : cardinal;
begin
    Result := Self.GetCRC;
end;


function TClipDataHelper.getPictureSize : string;
begin
    result := self.picturesize;
end;
procedure TClipDataHelper.setPictureSize(value : string);
begin
    self.picturesize := value;
end;
function TClipDataHelper.gethndIcon : THandle;
begin
    result := self.hndIcon;
end;
procedure TClipDataHelper.sethndIcon(value : THandle);
begin
    self.hndIcon := value;
end;
function TClipDataHelper.gettimestamp : TDateTime;
begin
    result := self.timestamp;
end;
procedure TClipDataHelper.settimestamp(value : TDateTime);
begin
    self.timestamp := value;
    self.timestampused := true;
end;
function TClipDataHelper.gettimestampused : Boolean;
begin
    result := self.timestampused;
end;
procedure TClipDataHelper.settimestampused(value : Boolean);
begin
    self.timestampused := value;
end;


//
//  TClipboardGrabber
//

const
    DELAY_MS = 50;
    MAX_DELAY_MS = 1000;


var crit : _RTL_CRITICAL_SECTION;
const LOCK_NAME = 'ArsClipClipboardLock';

class procedure TClipboardGrabber.InitLock;
begin
    //fLock := TMutex.Create();
    Windows.InitializeCriticalSection(crit);
end;

class procedure TClipboardGrabber.CreateLock;
var
    good : Boolean;
    lastErr : Cardinal;
    sleepcount : integer;
begin
    FrmDebug.AppendLog('CG: CreateLock');
    Windows.EnterCriticalSection(crit);
    FrmDebug.AppendLog('CG: Passed Lock');
end;
class procedure TClipboardGrabber.ReleaseLock;
begin
    FrmDebug.AppendLog('CG: ReleaseLock');
    Windows.LeaveCriticalSection(crit);
end;


class function TClipboardGrabber.hasFormat(format : Word) : boolean;
var i, err : integer;
const sleep_ms = 50;
begin
    repeat
        err := 0;
        Windows.SetLastError(ERROR_SUCCESS);
        Result := Windows.IsClipboardFormatAvailable(format);
        if not result then begin
            err := getLastError;
            if err <> 0 then begin
                sleep(sleep_ms);
                inc(i);
            end;
        end;
    until (err = 0) or ((i*sleep_ms) > 300);
end;
class function TClipboardGrabber.getLastDelay : integer;
begin
    result := fLastDelay;
end;
class function TClipboardGrabber.asText : string;
var
    success, alreadyOpen : boolean;
    tryCount : integer;

    function ClipboardAsText(var success : Boolean) : string;
    var h : THandle;
        p : pointer;
        sizeh : Cardinal;
    begin
        success := false;
        result := '';

        success := Windows.IsClipboardFormatAvailable(CF_UNICODETEXT);
        if success then begin
            h := Windows.GetClipboardData(CF_UNICODETEXT);
            success := h <> 0;
            if (success) then begin
                p := DupHandleToPointer(h,sizeh);
                if sizeh > 0 then begin
                    SetLength(result, sizeh div 2);
                    CopyMemory(@result[1], p, sizeh);
                    Result := string(PChar(Result));
                end;
                FreeMemory(p);
            end;

        end;
    end;
begin
    //
    // Try accessing the clipboard multipe times before erroring out.
    // This works around issues with programs that lock the clipboard for long
    // periods of time.
    //

    // NOTE: without "MySleep" and the ApplicationProcessMessages,
    // the clipboard would deadlock if another thread is accessing the clipboard
    //
    // TODO: use real multi-thread synch using a WaitObject command
    flastDelay := 0;
    success := false;
    result := '';
    tryCount := 0;

    alreadyOpen := fClipboardOpen;
    if not alreadyOpen then begin
        if not TryOpenClipboard then begin
            raise Exception.Create('Cannot open clipboard');
        end;
    end;

    result := ClipboardAsText(success);

    if not alreadyOpen then begin
        self.CloseClipboard;
    end;
end;
class function TClipboardGrabber.GetClipboardItem(
    clip : TClipItem; hi : HICON;
    OverrideFormat : WORD = 0;
    SizeLimit : cardinal = $FFFF
) : cardinal;
    procedure PreExit(log : string);
    begin
        FrmDebug.AppendLog(log, true);
        ReleaseLock;
    end;
    function GetFormat : TClipFormat;
    begin
        result := 0;
        try
            if OverrideFormat = 0 then begin
                result := ChooseAFormat;
            end else begin
                result := OverrideFormat;
            end;
        except
            FrmDebug.AppendLog('CG: <ClipItem - Find Format exception' + SysErrorMessage(GetLastError) );
        end;
    end;
    function SaveClipboardTo(ci : TClipItem; SizeLimit : cardinal; altStream : TMemoryStream = nil; altFormat : word = 0) : boolean;
    var
        CVolatileHandle : THandle;
        HasSizeLimit : boolean;
        pnt : Pointer;
    begin
        result := false;

        HasSizeLimit := SizeLimit <> $FFFF;
        if altFormat = 0 then begin
            CVolatileHandle := Windows.GetClipboardData(ci.GetFormat);
        end else begin
            CVolatileHandle := Windows.GetClipboardData(altFormat);
        end;
        if (CVolatileHandle = 0) then begin
            FrmDebug.AppendLog('CG: <ClipItem - can''t get handle 2> ', true);
            self.CloseClipboard;
            UnitMisc.MySleep(100);
            if altStream = nil then begin
                LastDitchAttempt(ci);
            end;
            EXIT;
        end;


        if altStream = nil then begin
            ci.ClearStream;
        end;
        pnt := unitmisc.DupHandleToPointer(CVolatileHandle, SizeLimit, HasSizeLimit);
        if (pnt = nil) then begin
            FrmDebug.AppendLog('CG: <ClipItem - can''t dup handle> ');
            self.CloseClipboard;
            LastDitchAttempt(ci);
            EXIT;
        end;

        if altStream = nil then begin
            ci.CData.size := SizeLimit;
            ci.GetStreamRaw_.Write(pnt^, ci.CData.size);
        end else begin
            altStream.Write(pnt^, SizeLimit);
        end;
        FreeMemory(pnt);
        result := true;
    end;
    function ClipboardAsText(var success : Boolean) : string;
    var h : THandle;
        p : pointer;
        sizeh : Cardinal;
    begin
        success := false;
        result := '';

        success := Windows.IsClipboardFormatAvailable(CF_UNICODETEXT);
        if success then begin
            h := Windows.GetClipboardData(CF_UNICODETEXT);
            success := h <> 0;
            if (success) then begin
                p := DupHandleToPointer(h,sizeh);
                SetLength(result, sizeh div 2);
                CopyMemory(@result[1], p, sizeh);
                Result := string(PChar(Result));
                FreeMemory(p);
            end;

        end;
    end;

    procedure HandledDisplayText(ci : TClipItem);
    var
        CVolatileHandle : THandle;
        sz : Cardinal;
        good : boolean;
        s : string;
        pnt : Pointer;
    begin
        ci.CData.picturesize_ := '';
        ci.CData.SetString('');

        if (ci.getformat = unitmisc.GetCF_HTML) then begin
            // richtext shadow for HTML clips
            if HasFormat(unitmisc.GetCF_RICHTEXT) then begin
                ci.RichShadowMemory_ := TMemoryStream.Create;
                if SaveClipboardTo(ci, SizeLimit, TMemoryStream(ci.RichShadowMemory_), UnitMisc.GetCF_RICHTEXT) then begin
                    FrmDebug.AppendLog('CG: <ClipItem - RichText Shaddow');
                end else begin
                    ci.RichShadowMemory_.Free;
                    ci.RichShadowMemory_ := nil;
                end;
                {
                CVolatileHandle := Windows.GetClipboardData(unitmisc.GetCF_RICHTEXT);
                if (CVolatileHandle <> 0) then begin
                    pnt := unitmisc.DupHandleToPointer(CVolatileHandle, sz);
                    ci.RichShadowMemory_.Write(pnt^, sz);
                    ci.RichShadowMemory_.Position := 0;
                    FreeMemory(pnt);
                    ci.GetRichTextShadow(s);
                    s := CleanRichtextURLs(s);
                    TMemoryStream(ci.RichShadowMemory_).Clear;
                    ci.RichShadowMemory_.Write(s[1], length(s) * sizeof(char));
                    ci.RichShadowMemory_.Position := 0;


                    FrmDebug.AppendLog('<ClipItem - RichText Shaddow');
                end;
                }
            end;
            s := ClipboardAsText(good);
            if good then begin
                ci.CData.SetString(s);
            end;
        end else if (ci.getformat = unitmisc.GetCF_RICHTEXT) then begin
            ci.RichShadowMemory_ := TMemoryStream.Create;
            if SaveClipboardTo(ci, SizeLimit, TMemoryStream(ci.RichShadowMemory_), UnitMisc.GetCF_RICHTEXT) then begin

                FrmDebug.AppendLog('CG: <ClipItem - RichText Shaddow');
                s := ClipboardAsText(good);
                if good then begin
                    ci.CData.SetString(s);
                end;
            end else begin
                ci.RichShadowMemory_.Free;
                ci.RichShadowMemory_ := nil;
            end;

            {
            ci.RichShadowMemory_ := TMemoryStream.Create;
            ci.RichShadowMemory_.Position := 0;

            ci.GetRichText(s);
            s := CleanRichtextURLs(s);

            TMemoryStream(ci.RichShadowMemory_).Clear;
            ci.RichShadowMemory_.Write(s[1], length(s) * sizeof(char));
            ci.RichShadowMemory_.Position := 0;

            FrmDebug.AppendLog('<ClipItem - RichText Shadow>');
            s := ClipboardAsText(good);
            if good then begin
                ci.CData.SetString(s);
            end;   }
        end else if (ci.getformat = CF_HDROP) then begin
            CVolatileHandle := Windows.GetClipboardData(CF_HDROP);
            s := ci.GetFilenamesAsText_(CVolatileHandle) ;
            FrmDebug.AppendLog('CG: filenames: ' + s);
            ci.CData.SetString(s);
        end else if (ci.getFormat = CF_DIB) then begin
            ci.cdata.picturesize_ := ci.GetFormatName(true);
        end else  if ci.getFormat <> CF_UNICODETEXT then begin
            s := ClipboardAsText(good);
            if good then begin
                ci.CData.SetString(s);
            end;
        end;
    end;
    procedure HandleSizeWorkarounds(ci : TClipItem);
    var
        Estimation : cardinal;
    begin
        // vba workaround - garbage in the clip
        if (ci.getFormat = CF_UNICODETEXT) and (ci.cdata.size = 2048) then begin
            Estimation := ((length(ci.GetAsPlaintext)+1)*sizeof(Char));
            if Estimation <> ci.Cdata.size  then begin
                ci.cdata.size := Estimation;
            end;
        end;
        // workaround for a terminal program, might need to generalize this
        // more for other programs instead of a sentinal value
        if (ci.getFormat = CF_UNICODETEXT) and (ci.cdata.size = $4000) {16,384} then begin
            Estimation := ((length(ci.GetAsPlaintext)+1)*sizeof(Char));
            if Estimation <> ci.Cdata.size  then begin
                ci.cdata.size := Estimation;
            end;
        end;
    end;

begin
    FrmDebug.AppendLog('CG: <ClipItem> ', true);
    CreateLock;
    Windows.SetLastError(ERROR_SUCCESS);     // workaround for some OS's

    clip.setformat(0);
    clip.cdata.Size := 0;
    if (hi <> 0) then begin
        clip.CData.SetHICON(hi, UnitMisc.IconCRC(hi));
    end;

//    clip.CData.hndIcon_ := hi;
    clip.CData.SetString('');
    clip.CData.displaytext := '';
    result := 0;


    clip.ClearStream;
    clip.setFormat(GetFormat);
    if (clip.getFormat  = 0) then begin
        PreExit('CG: <ClipItem - No Supported Format ');
        EXIT;
    end;
    try
        if not TryOpenClipboard then begin
            Application.ProcessMessages;
            LastDitchAttempt(clip);
            if (clip.GetFormat = 0) then begin
                PreExit('CG: <ClipItem - can''t open clipboard2 > ');
                EXIT;
            end;
        end else begin
            if not SaveClipboardTo(clip, SizeLimit) then begin
                PreExit('CG: SaveClipboardTo failed');
                self.CloseClipboard;
                EXIT;
            end;
        end;
        HandledDisplayText(clip);
        self.CloseClipboard;
        HandleSizeWorkarounds(clip);
        clip.CData.Hash := clip.GetCRC_;
    except
        on E: Exception do begin
            self.CloseClipboard;
            clip.setFormat(0);
            result := 0;

            PreExit('CG: <clipItem Dup Exception - ' + E.Message + ' ');
            EXIT;
        end;
    end;

    PreExit('CG: <ClipItem ClipboardSave success!> size=' + IntToSTr(clip.CData.size) );
    clip.cdata.timestamp_ := now;
    result := clip.getFormat;

end;
class function TClipboardGrabber.ClipboardHasPicAndText(var textformat :word) : boolean;
var cfl : TList<TClipFormat>;
    i : integer;
begin
    cfl := GetAvailableFormats;
    result := cfl.Contains(CF_DIB);
    result := result and (
        cfl.Contains(CF_UNICODETEXT) or
        cfl.Contains(UnitMisc.GetCF_HTML) or
        cfl.Contains(UnitMisc.GetCF_RICHTEXT) or
        cfl.Contains(CF_TEXT)
    );
    result := result and (TClipboardGrabber.asText <> '');

    if (result) then begin
        textFormat := GetConfiguredTextFormat(cfl);
    end;
    myfree(cfl);
end;

class function TClipboardGrabber.GetConfiguredTextFormat(listOrNil : TClipFormatList) : TClipFormat;
var i : integer;
begin
    if listOrNil = nil then begin
        listOrNil := GetAvailableFormats;
    end;
    result := 0;
    for i := 0 to FrmConfig.FormatCount - 1 do begin
    case FrmConfig.GetFormat(i) of
    FO_RICHTEXT:
        begin
            if listOrNil.Contains(UnitMisc.GetCF_RICHTEXT) then begin
                result := UnitMisc.GetCF_RICHTEXT;
                BREAK;
            end
        end;
    FO_HTML:
        begin
            if listOrNil.Contains(UnitMisc.getCF_HTML) then begin
                result := UnitMisc.GetCF_HTML;
                BREAK;
            end
        end;
    end;
    end;

    if result = 0 then begin
        if listOrNil.Contains(CF_UNICODETEXT) then begin
            result := CF_UNICODETEXT;
        end else begin
            result := CF_TEXT;
        end;
    end;
end;
class function TClipboardGrabber.GetAvailableFormats : TClipFormatList;
var cf : TClipFormat;
    err : integer;
begin
    Windows.SetLastError(ERROR_SUCCESS);
    result := TList<TClipFormat>.Create;
    Sleep(10);
    // NOTE: I have no idea why, but all results will fail with no LastError code
    // in certain instances unless there's a sleep first.
    // Windows 10 SNIP TOOL is an example

    if HasFormat(CF_TEXT) then result.Add(CF_TEXT);
    if HasFormat(CF_WAVE) then result.Add(CF_WAVE);
    if HasFormat(CF_DIB) then result.Add(CF_DIB);
    if HasFormat(CF_HDROP) then result.Add(CF_HDROP);
    if HasFormat(UnitMisc.GetCF_RICHTEXT) then result.Add(UnitMisc.GetCF_RICHTEXT);
    if HasFormat(UnitMisc.getCF_HTML) then result.Add(UnitMisc.getCF_HTML);
    if HasFormat(CF_UNICODETEXT) then result.Add(CF_UNICODETEXT);
end;

class function TClipboardGrabber.TryOpenClipboard : boolean;
var i : integer;
    openOK : boolean;
    errmsg : string;
begin
    if fClipboardOpen then begin
        FrmDebug.AppendLog('CG: TryOpenClipboard: already open, exiting.');
        Exit;
    end;

    flastDelay := 0;

    // Try a few times to open the the clipboard
    i := 0;
    errmsg := '';
    while (i*DELAY_MS < MAX_DELAY_MS) do begin
        SetLastError(ERROR_SUCCESS);
        result := TMyClipboard.OpenClipboard(Application.Handle,'tryopenclipboard');
        if result then begin
            fClipboardOpen := true;
            BREAK;
        end;

        errmsg := 'CG: <ClipItem - can''t open clipboard> attempt = '+IntToStr(i);

        Sleep(DELAY_MS);
        inc(i);
    end;


    if (i > 0) then begin
        if result then begin
            FrmDebug.AppendLog('CG: <ClipItem - open> attemps =' +IntToStr(i));
        end else begin
            FrmDebug.AppendLog(errmsg, true);
        end;
        flastDelay := i*DELAY_MS;
    end;

end;
class procedure TClipboardGrabber.CloseClipboard;
begin
    if not fClipboardOpen then FrmDebug.AppendLog('CG: CloseClipboard called when not Open');

    if fClipboardOpen then begin
        fClipboardOpen := False;
        TMyClipboard.CloseClipboard;
    end;
end;

class function TClipboardGrabber.ChooseAFormat : TClipFormat;
var i : integer;
    cfl : TList<TClipFormat>;
    HasPicture : boolean;
begin
    result := 0;
    cfl := GetAvailableFormats;
    if cfl.Count = 0 then begin
        myfree(cfl);
        EXIT;
    end;


//    HasText := cfl.Contains(CF_TEXT);
    HasPicture := cfl.Contains(cf_DIB);


    if cfl.Contains(CF_WAVE) then begin
        result := CF_WAVE;
    end else if cfl.Contains(CF_DIB) then begin
        result := CF_DIB;
    end else if cfl.Contains(CF_HDROP) then begin
        result := CF_HDROP;
    end;

    if result = 0 then result := GetConfiguredTextFormat(cfl);

    myfree(cfl);
end;
class procedure TClipboardGrabber.LastDitchAttempt;
var s : string;
begin
    ci.setFormat(0);
    s := TClipboardGrabber.asText;
    if s <> '' then begin
        ci.CData.SetString(s);
        ci.setFormat(CF_TEXT);
    end;
end;
class function TClipboardGrabber.CleanRichtextURLs(s : string) : string;
var str : string;
    astr : ansistring;
begin
    result := str;
    EXIT;

    // TODO - research this more.
    // it seems to fail more than succeed

    str := pansichar(@s[1]);
   // remove the link between the quotes
    astr := '';
    while str <> '' do begin
        astr := astr + UnitToken.TokenString(str,'fldinst HYPERLINK "',false);
        if str = '' then begin
            astr := astr + #0;
            setlength(str, length(astr) div 2);
            move(astr[1], str[1], length(astr));
            result := str;
            EXIT;
        end;

        UnitToken.TokenString(str,'"}',false);
        astr := astr + 'fldsint HYPERLINK ""}';
    end;
end;

initialization
begin
    TClipboardGrabber.InitLock;
    //WIndows.InitializeCriticalSection(crit);
end


finalization
begin
    TFrmDebug.MeOnlyAppend('UnitClipboardGrabber', false);
end;


end.
