unit UnitJS;

interface

uses UnitPaste, System.Classes, Windows;

type TJSCompleteCallback = procedure of object;
type
TJavaScript = class(TObject)
    protected
        fRequiresFocus : boolean;
        lastErr,errLog : string;
        fGeneratedScript : string;
        isConnected, isCompleted : boolean;
        ScriptComplete : boolean;
        signalFilename : string;
        pipeServer : TThread;
        scriptWatcher : TThread;
        AnyCallbackComplete : boolean;
        crit : _RTL_CRITICAL_SECTION;

        function createJSLitteralString(const str : string) : string;
        function MacroToJS(const macro : string) : string;
        function executeJSTask(hnd : THandle; paste : TPaste; javascript : string) : boolean;
        class function getFileHeader(pipeID : integer) : string;


        procedure ReadConnectedCallback(h : THandle);
        procedure TimeoutCallback(cmdResult : integer; errString : string);
        procedure ScriptCompleteCallback;
        procedure ReadErrorCallback(errMessage : string);
    public
        constructor Create;
        destructor Destroy; override;
        class function getStartMacro : string;
        class function getEndMacro : string;
        class function getStartNoMacro : string;
        class function getEndNoMacro : string;
        class function isJavaScript(str : string) : boolean;
        class function GetMainLineOffset(scriptContent : string) : integer;
        class function GetScriptName : string;
        class function getMacroString : string;

        function requiresFocus : boolean;
        function getLastErr : string;
        function executeJS(hnd : THandle; paste : TPaste; javascript : string) : boolean;
        property GeneratedScript : string read fGeneratedScript;
end;





implementation

uses UnitToken, SysUtils, StrUtils,  UnitMisc, UnitFrmDebug, IOUtils,
 Forms, VCL.Clipbrd, Dialogs, UnitClipQueue;


// The user will only see this function
const
JS_NL = #13#10;
JS_START_MACRO = 'start_ac_macro:'#13#10;
JS_END_MACRO = 'end_ac_macro:'#13#10;

JS_POPUP_CLIPS = 'popupClips';
JS_SET_POPUP_CLIPS = '//@set '+JS_POPUP_CLIPS+'[]=';
JS_SET_ON = '"on"';
JS_SET_OFF = '"off"';
JS_MAIN_FUNCTION = 'function main(clipboardStr)';
JS_MAIN_HEADER =
    JS_MAIN_FUNCTION + ' {' + JS_NL +
    JS_SET_POPUP_CLIPS + JS_SET_OFF + JS_NL +
    '//change to "on" to include the Popup Clips list as a string array' + JS_NL +
    '//' + JS_NL +
    '// use manuallyExecuteMacro(string) for dynamically built macros' + JS_NL +
    '// use saveToPopupClips(string) to save text to the Popup Clips list' + JS_NL +
    '// use getCurrentClipboard() to get the current state of the clipboard'+JS_NL;
JS_MAIN_FOOTER =
    '}';
JS_MAIN_HEADER_W_MACRO =
    JS_MAIN_HEADER +
    JS_START_MACRO;
JS_MAIN_FOOTER_W_MACRO =
    JS_END_MACRO + JS_MAIN_FOOTER;
JS_SCRIPTNAME = 'main.js';


class function TJavaScript.isJavaScript(str : string) : boolean;
var i : integer;
begin
    i := ansiPos(JS_MAIN_FUNCTION, str);
    result := (i > 0);
end;

class function TJavaScript.getStartMacro : string;
begin
    result := JS_MAIN_HEADER_W_MACRO;
end;
class function TJavaScript.getEndMacro : string;
begin
    result := JS_MAIN_FOOTER_W_MACRO;
end;
class function TJavaScript.getMacroString : string;
begin
    result := getStartMacro + getEndMacro;
end;
class function TJavaScript.getStartNoMacro : string;
begin
    result := JS_MAIN_HEADER;
end;
class function TJavaScript.getEndNoMacro : string;
begin
    result := JS_MAIN_FOOTER;
end;




const PIPE_NAME = 'ACJavaScript';

function getPipeName(pipeID : integer) : string;
begin
    result := '\\.\pipe\'+PIPE_NAME+IntToStr(pipeId);
end;

//
// TClientSingal tell the client
// note: poorman's IPC using atomic file create operation
//
type TClientSignal = class(TObject)
    const
        JS_SIGNALNAME = 'signal.txt';

    public
        class procedure Send(value : boolean);
        class function getName : string;
end;
class procedure TClientSignal.Send(value: Boolean);
var
    s : string;
    tf : TextFile;
begin
    s := TClientSignal.getName;
    case value of
    true:
        begin
            AssignFile(tf, s);
            Rewrite(tf);
            CloseFile(tf);
        end;
    false:
        begin
            if FileExists(s) then
                SysUtils.DeleteFile(s);
        end;
    end;
end;
class function TClientSignal.getName;
begin
    result :=  TPath.Combine(GetAppPath, 'JavaScript\' +JS_SIGNALNAME);
end;


//
// PipeServer
// - ends when an Error or when Connected
//
type TPipeDataMode = (PIPESERVER_READONLY, PIPESERVER_WRITEONLY);
type TErrorCallback = procedure (s : string) of object;
type TPipeConnectedCallback = procedure (pipeH : THandle) of object;
type TPipeServer = class(TThread)
    private
        pipeID : integer;
        pipeH : THandle;
        dataMode : TPipeDataMode;
        fConnectedCallback : TPipeConnectedCallback;
        fErrorCallback : TErrorCallback;
        attempingConnect : boolean;
    public
        procedure Init(pipeID : integer; dataMode : TPipeDataMode);
        procedure Execute; override;
        procedure InteruptConnect;
        property ConnectedCallback : TPipeConnectedCallback read fConnectedCallback write fConnectedCallback;
        property ErrorCallback : TErrorCallback read fErrorCallback write fErrorCallback;
end;
procedure TPipeServer.Init(pipeID : integer; dataMode : TPipeDataMode);
begin
    self.pipeID := pipeID;
    self.dataMode := dataMode;
end;
procedure TPipeServer.Execute;
    function GetPipeHandle(var h : THandle; pipeID : integer) : boolean;

        function createPipe(name : string) : THandle;
        var
            s : string;
            mode : integer;
        const
            MAX_BUFFERS = 100;
            KILOBYTES = 1024;
            MAX_SECONDS = 10;
            MILLISECONDS = 1000;
            MAX_INSTANCES = 10;

        begin
            s :=  name;
            case dataMode of
            PIPESERVER_READONLY: mode :=   PIPE_ACCESS_DUPLEX;   // Yes, this is dumb but needed
            PIPESERVER_WRITEONLY: mode := PIPE_ACCESS_DUPLEX;
            end;

            result := CreateNamedPipe(
                PChar(S),
                mode Or FILE_FLAG_WRITE_THROUGH {or FILE_FLAG_OVERLAPPED},
                PIPE_WAIT Or PIPE_TYPE_MESSAGE or PIPE_READMODE_MESSAGE, // read/write mode
                MAX_INSTANCES,
                MAX_BUFFERS * KILOBYTES, MAX_BUFFERS * KILOBYTES, // out and in buffers
                MAX_SECONDS * MILLISECONDS,  // client timeout
                nil
            );
        end;
    var errCount : integer;
    begin
        errCount := 0;
        repeat
            h := createPipe(getPipeName(pipeId));
            if (h = INVALID_HANDLE_VALUE) then begin
                inc(errCount);
                MySleep(10);
            end;
        until (h<> INVALID_HANDLE_VALUE) or  (errCount >= 5);
        result := h <> INVALID_HANDLE_VALUE;
    end;
    function PerformPipeConnect(h : THandle) : boolean;
    begin
        result := ConnectNamedPipe(h, nil);
    end;
var
    eventHandle : THandle;
begin
    attempingConnect := false;
    pipeH := 0;
    if not GetPipeHandle(pipeH, pipeID) then begin
        if assigned(fErrorCallback) then begin
            fErrorCallback('create named pipe failed');
        end;
        EXIT;
    end;

    attempingConnect := true;
    if (PerformPipeConnect(pipeH)) then begin
        if not Terminated and assigned(fConnectedCallback)  then begin
            fConnectedCallback(pipeH);
        end else begin
            DisconnectNamedPipe(pipeH);
            CloseHandle(pipeH);
        end;
    end else begin
        if not Terminated and assigned(fErrorCallback) then begin
            fErrorCallback('connect named pipe failed');
        end;
    end;
end;
procedure TPipeServer.InteruptConnect;
var s : string;
    tf : TextFile;
    h : THandle;
    bytes : cardinal;
const
    END_MESSAGE = 'END';
begin
    self.Terminate;
    s := getPipeName(pipeId);
    try
        h := CreateFile(
            PWideChar(s),
            GENERIC_WRITE,FILE_SHARE_WRITE,
            nil,
            OPEN_EXISTING, 0, 0
        );
		if(h <> INVALID_HANDLE_VALUE) then begin
            if not WriteFile(h, END_MESSAGE, length(END_MESSAGE)*SizeOf(char), bytes, nil) then begin
                FrmDebug.AppendLog('Cant intertupt PipeServer', true);
            end;
			CloseHandle(h);
		end else begin
            FrmDebug.AppendLog('Cant open pipe to intertupt', true);
        end;
    except
    end;
end;

//
// ScriptWatcher
// - ends when script is Complete or Times out
//
type TScriptCompleteCallback = procedure of object;
type TScriptTimeoutCallback = procedure(cmdResult : integer; errMessage : string) of object;
type TScriptWatcher = class(TThread)
    private
        cmdResult : integer;
        fTimeoutCallback : TScriptTimeoutCallback;
        fCompleteCallback : TScriptCompleteCallback;
    public
        procedure init(cmdResult : integer);
        procedure Execute; override;
        property TimeoutCallback : TScriptTimeoutCallback read fTimeoutCallback write fTimeoutCallback;
        property ScriptCompleteCallback : TScriptCompleteCallback read fCompleteCallback write fCompleteCallback;
end;
procedure TScriptWatcher.init(cmdResult : integer);
begin
    self.cmdResult := cmdResult;
end;
procedure TScriptWatcher.Execute;
    var
        lastErr : string;
    function WaitForTerminate(cmdResult : integer) : boolean;
    begin
        result := false;
        if  WaitForSingleObject(cmdResult, 100) <> WAIT_OBJECT_0 then begin
            case WaitForSingleObject(cmdResult, 2000) of
            WAIT_FAILED :
                begin
                    lastErr := SysErrorMessage( GetLastError );
                end;
            WAIT_TIMEOUT:
                begin
                    //lastErr := '';
                end;
            WAIT_OBJECT_0:
                begin
                    result := true;
                end;
            end;
        end else begin
            result := true;
        end;
        if result then begin
            if not CloseHandle(cmdResult) then begin
    //            FrmDebug.AppendLog('JavaScript: couldn''t close process handle', true);
            end;
        end;
    end;
begin
    if not WaitForTerminate(cmdResult) then begin
        if Terminated then EXIT;
        fTimeoutCallback(cmdResult, lastErr);
    end else begin
        if Terminated then EXIT;
        fCompleteCallback;
    end;
end;

// the script will write AC Macros to the named pipe for communication

const NL = JS_NL;

// insert statically instantiated objects here
//    function main() {
JF_MAIN_FOOTER =
//    '} // end main()
    'main(clipboardStr);' +  NL +
    'pipe.Write("END");' + NL +
    'pipe.Close();' +
    'WScript.Quit(1)';




constructor TJavaScript.Create;
begin
    InitializeCriticalSection(crit);
end;
destructor TJavaScript.Destroy;
begin
    DeleteCriticalSection(crit);
    inherited Destroy;
end;
class function TJavaScript.getFileHeader(pipeID : integer)  : string;
var s : string;
    clip : string;
    signal : string;
const
    cliptext = 'clip.txt';
begin
    s := TPath.Combine(GetAppPath, 'JavaScript');
    clip := TPath.Combine(s, cliptext);
    signal := TClientSignal.getName;
    s := ReplaceStr(s, '\','\\');
    clip := ReplaceStr(clip, '\','\\');
    signal := ReplaceStr(signal, '\','\\');

    result :=  'var fs = new ActiveXObject("Scripting.FileSystemObject");' + NL +
    'var pipe = fs.CreateTextFile("\\\\.\\pipe\\'+PIPE_NAME+IntToStr(pipeId)+'", false);' + NL +
    'function _waitForSignal() {'+NL+
    ' while (!fs.FileExists("'+signal+'")) {};'+NL+
    ' fs.DeleteFile("'+signal+'");'+NL+
    '}'+NL+
    'function _pipeWrite(s) {' +NL+
    ' pipe.Write(s);'+NL+
    ' _waitForSignal();'+NL+
    '};'+NL+
    'function manuallyExecuteMacro(macro) {_pipeWrite(macro)}' + NL +
    'function saveToPopupClips(clip) {_pipeWrite(''[newclip=\"''+clip.replace(''"'',''""'')+''\"]'');}' +NL +
    'function getCurrentClipboard() {' +NL+
    '  var s = "'+clip+'";'+NL+
    '  if (fs.FileExists(s)){ fs.DeleteFile(s); }'+NL+
    '  _pipeWrite("[TOTEXTFILE=\""+s+"\"]");' +NL+
    '  var cl = fs.OpenTextFile(s, 1, false, -1);' + NL +
    '  var result = cl.ReadAll();'+NL+
    '  cl.close();'+NL+
    '  return result;'+NL+
    '}';
end;



function TJavaScript.getLastErr : string;
begin
    result := lastErr;
end;
function TJavaScript.createJSLitteralString(const str : string) : string;
begin
    result := str;
    if length(result) = 0 then begin
        result := '""';
        EXIT;
    end;

    if result[length(result)] = #0 then begin
        delete(result,length(result),1);
    end;
    result := ReplaceText(result, '\', '\\');
    result := ReplaceText(result, '''', '\''');
    result := ReplaceText(result, '"', '\"');
    result := ReplaceText(result, #13, '\r');
    result := ReplaceText(result, #10, '\n');
    result := '"' + result + '"';
end;
function TJavaScript.MacroToJS(const macro : string) : string;
var
    s : string;
begin
    s := macro;
    if macro[length(macro)] = #10 then begin
        delete(s,length(s)-1, 2);
    end;
    s := createJSLitteralString(s);
    result := 'pipe.Write('+ s +');' + NL;
end;
function TJavaScript.executeJS(hnd : THandle;paste : TPaste; javascript : string) : boolean;
begin
    //
    // block while threads are working to make calls to this function
    // synchonous
    //
    lastErr := '';
    executeJSTask(hnd, paste, javascript);
    while not AnyCallbackComplete do begin
        mysleep(50);
    end;

    result := lastErr = '';
end;
function TJavaScript.executeJSTask(hnd : THandle;paste : TPaste; javascript : string) : boolean;
    procedure SaveMainScript(var path : string; pipeID : integer);
        function encodeACMacros(javascript : string) : string;
        var
            s : string;
            macro : string;
        begin
            while (javascript <> '') do begin
                result := result + TokenString(javascript,JS_START_MACRO,false);
                if (javascript <> '') then begin
                    macro := TokenString(javascript,JS_END_MACRO,false);
                    result := result + macroToJS(macro);
                end;
            end;
        end;
        function wrapEncodedMacros(encodedMacros : string;pipeID : integer) : string;
        var
            i : integer;
            cnt : integer;
            clipb : string;
        begin
            result := '';
            clipb := Clipboard.AsText;

            cnt := ClipQueue.GetQueueCount;
            if PosI(JS_SET_POPUP_CLIPS+JS_SET_ON, encodedMacros) <> 0 then begin
                result := 'var '+JS_POPUP_CLIPS+'= [' ;
                for i:= 0 to cnt - 1  do begin
                    result := result + self.createJSLitteralString(ClipQueue.GetClipItem(i).GetAsPlaintext);
                    if (i<>(cnt - 1)) then begin
                        result := result + ',';
                    end;
                end;
                result := result + '];' + JS_NL;
            end;
            result := result +

            getFileHeader(pipeID) +
            'var clipboardStr = '+ createJSLitteralString(clipb) + ';'+ JS_NL +
            encodedMacros + JS_NL +
            JF_MAIN_FOOTER;
        end;
    var
        sl : TStringList;
    begin
        fGeneratedScript := encodeACMacros(javascript);
        fGeneratedScript := wrapEncodedMacros(fGeneratedScript, pipeID);

        path := TPath.Combine(GetAppPath, 'JavaScript');
        if not DirectoryExists(path) then begin
            ForceDirectories(path);
        end;

        path := TPath.Combine(path, JS_SCRIPTNAME);
        sl := TStringList.Create;
        sl.Add(fGeneratedScript);
        sl.SaveToFile(path);
        myfree(sl);
    end;
    procedure InitErrorLog(var errLog : string);
    begin
        errLog := TPath.Combine(GetAppPath,'JavaScript\err.txt');
        DeleteFile(PChar(errLog));
    end;
    function IsRunning(cmdResult : integer) : boolean;
    var
        r : cardinal;
    begin
        result := false;
        if GetExitCodeProcess(cmdResult, r) then begin
            result := r = STILL_ACTIVE;
        end;
    end;
var
    h : THandle;
    command: string;
    cmdResult : integer;
    path : string;
    pipeID : integer;
    eventHandle : THandle;
    r : cardinal;
    tm : cardinal;
    ps : TPipeServer;
    sw : TScriptWatcher;
begin
    tm := Windows.GetTickCount;
    frmDebug.AppendLog('JavaScript started');
    result := false;
    fGeneratedScript := '';
    pipeID := Random(10000);
    AnyCallbackComplete := false;

    SaveMainScript(path, pipeID);

    isConnected := false;
    ScriptComplete := false;

    ps :=  TPipeServer.Create(true);
    pipeServer := ps;
    ps.Init(pipeID, PIPESERVER_READONLY);
    ps.ConnectedCallback := ReadConnectedCallback;
    ps.ErrorCallback := ReadErrorCallback;
    ps.FreeOnTerminate := true;
    ps.Start;

    TClientSignal.Send(false);
    InitErrorLog(errLog);

    command := 'cmd.exe /s /c "cscript /nologo "'+path+'" 2> "'+errLog+'""';
    cmdResult := UnitMisc.RunCommandLine(command);

    if (RunCommandResultSuccess(cmdResult)) then begin
        sw := TScriptWatcher.Create(true);
        scriptWatcher := sw;
        sw.init(cmdResult);
        sw.TimeoutCallback := TimeoutCallback;
        sw.ScriptCompleteCallback := ScriptCompleteCallback;
        sw.FreeOnTerminate := true;
        sw.Start;
    end else begin
        FrmDebug.AppendLog(
            SysErrorMessage(RunCommandResultToSysErrorCode(cmdResult))
        );
        lastErr := 'Error: JavaScript command failed to run';
    end;
    FrmDebug.AppendLog('JavaScript end');
    result := false; //
end;

//
// callbacks
// Normal execution will trigger a ReadConnected and ScriptComplete callback (mutually exclusive execution)
//
procedure TJavaScript.ReadConnectedCallback(h : THandle);
    function PerformPipeReads(h : THandle) : boolean;
    var
        connectOK : boolean;
        byteCount : DWORD;
        bytesRead, bytesWritten : Cardinal;
        buffer : array[0..1000*100] of byte;
        inMacro : string;
        endFound : boolean;
    begin
        result := false;
        connectOK := true;
        repeat
            inMacro := '';
            if ReadFile(h,buffer,sizeof(buffer),bytesRead,nil) then begin
                TClientSignal.Send(false);
                buffer[bytesRead] := 0;
                inMacro := inMacro + string(pansichar(@buffer));
                FrmDebug.AppendLog('Pipe Read',false);
                connectOK := true;
            end else begin
                FrmDebug.AppendLog('Pipe Read Failed',false);
                connectOK := false;
                BREAK;
            end;

            endFound := (inMacro = 'END');
            if not endFound then begin
                Paste.SendMacro(KEYS_STR+inMacro);
            end;
            TClientSignal.Send(true);;
        until endFound;
        result := connectOK;
    end;
begin
    try
        // the Script watcher is responsible for setting AnyCallbackComplete
        EnterCriticalSection(crit);
        isConnected := true;

        FrmDebug.AppendLog('ReadConnectedCallback');
        if PerformPipeReads(h) then begin
            ScriptComplete := true;
//            if not IsCompleted then
//                scriptWatcher.Terminate;
        end else begin
            if lastErr = '' then begin
                lastErr := 'Error: unable to read from pipe';
            end;
        end;

        DisconnectNamedPipe(h);
        CloseHandle(h);
    finally
        LeaveCriticalSection(crit);
    end;
end;
procedure TJavaScript.ReadErrorCallback(errMessage : string);
begin
    FrmDebug.AppendLog('ReadErrorCallback: '+errMessage);
    AnyCallbackComplete := true;
    //
    // let the script time out on its own
    //
end;

procedure TJavaScript.ScriptCompleteCallback;
    procedure PerformErrorReporting(errLog : string);
    var
        sl : TStringList;
        s : string;
        normalClose : boolean;
    begin
        // Wait more time only if the script is still running
        sl := TStringList.Create;
        sl.LoadFromFile(errLog);
        s := sl.Text;
        if (s<>'') then begin
            frmdebug.AppendLog(s, false);
        end;
        lastErr := s;
        myfree(sl);
    end;
begin
    // Three cases
    // A script may never run, may terminate with an error, or may complete normally

    try
        EnterCriticalSection(crit);
        isCompleted := true;
        FrmDebug.AppendLog('ScriptCompleteCallback');

        if isConnected then begin
            if not ScriptComplete then begin
                PerformErrorReporting(errLog);
                if lastErr = '' then lastErr := 'Error: unexpected script termination';
            end else begin
                FrmDebug.AppendLog('ScriptCompleteCallback normal complete');
            end;
        end else begin
            PerformErrorReporting(errLog);
            TPipeServer(pipeServer).InteruptConnect;
            if lastErr = '' then lastErr := 'Error: unexpected script termination';
        end;
    finally
        AnyCallbackComplete := true;
        LeaveCriticalSection(crit);
    end;
end;
procedure TJavaScript.TimeoutCallback(cmdResult : integer; errString : string);
var c : cardinal;
begin
    FrmDebug.AppendLog('TimeoutCallback');
    if errString = '' then lastErr := 'Error: Script timed out';

    if not isConnected then begin
        TPipeServer(pipeServer).InteruptConnect;
        try
            TerminateProcess(cmdResult, c);
            CloseHandle(cmdResult);
        except
        end;
    end;

    AnyCallbackComplete := true;
end;


class function TJavaScript.GetMainLineOffset(scriptContent : string) : integer;
var
    s : string;
    i : integer;
begin
    result := -1;
    i := 0;
    repeat
        s := UnitToken.TokenString(scriptContent, JS_NL, false);
        if pos(JS_MAIN_FUNCTION, s) > 0 then begin
            result := i;
            BREAK;
        end;
        inc(i);
    until scriptContent = '';
end;
class function TJavaScript.GetScriptName : string;
begin
    result := JS_SCRIPTNAME;
end;

function TJavaScript.requiresFocus : boolean;
begin
    result := fRequiresFocus;
end;

initialization
begin
end;
end.
