unit UnitJScript;

interface


uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, AscrLib, ActiveX, ComObj, StdCtrls, ObjComAuto, unitPaste;


type TJScript = class(TComponent, IActiveScriptSite)
   private
        fScript: IActiveScript;
        fParse: IActiveScriptParse;
        terminated : boolean;
        fPaste : TPaste;
        fLastErr : string;
        fLastLine : Cardinal;
        fLastPos : LongInt;

        // IActiveScriptSite interface stuff
        function  GetLCID(out plcid: LongWord): HResult; stdcall;
        function  GetItemInfo(pstrName: PWideChar;
                              dwReturnMask: LongWord;
                              out ppiunkItem: IUnknown;
                              out ppti: IUnknown): HResult; stdcall;
        function  GetDocVersionString(out pbstrVersion: WideString): HResult; stdcall;
        function  OnScriptTerminate(var pvarResult: OleVariant;
                                    var pexcepinfo: EXCEPINFO): HResult; stdcall;
        function  OnStateChange(ssScriptState: tagSCRIPTSTATE): HResult; stdcall;
        function  OnScriptError(const pscripterror: IActiveScriptError): HResult; stdcall;
        function  OnEnterScript: HResult; stdcall;
        function  OnLeaveScript: HResult; stdcall;

        //
        procedure RunScript(script : string);
        function getLastErr : string;

    public
        constructor Create();

        class function isJavaScript(str : string) : boolean;
        class function getStartMacro : string;
        class function getEndMacro : string;
        class function getStartNoMacro : string;
        class function getEndNoMacro : string;

        function executeJS(paste : TPaste; javascript : string) : boolean;

        property LastErr : string read getLastErr;
        property LastLine : cardinal read fLastLine;
        property LastPos : integer read fLastPos;
end;

implementation

uses Clipbrd, UnitClipQueue, System.StrUtils, UnitToken, UnitClipboardGrabber,
unitMisc, UnitFocusManager,
UnitFrmDebug, UnitFrmPermanentNew, UnitUIAutomation, UnitMyClipboard;

const
    CLSID_JScript: TGUID = '{f414c260-6ac0-11cf-b6d1-00aa00bbbb58}';
//    CLSID_JScript: TGUID = '{16d51579-a30b-4c8b-a276-0ff4dc41e755}'; // Chakra Engine (ie9)
    SCRIPT_E_REPORTED = HRESULT($80020101);
    ArsClipObject = '_ArsClip';


// NOTES: For some reason, functions do not allow parameters
// Parameters are set separately (assembly style) and then
// the function is called.
// Wrapper functions are used to hide these details in the JavaScript

{$M+}
type TScriptObject = class(TComponent)
    private
        fPaste : TPaste;
        findex : integer;
        fHandle : integer;
        fPinnedIndex : Integer;
        fTitle : string;
        fFormat : string;

        fclientName : string;
        fclientX : integer;
        fclientY : integer;
        fclientPath : string;

        fDebugText : string;

        function getClipboard : string;
        procedure setClipboard(value : string);

        function getPopupClip : string;
        procedure setPopupIndex(value : Integer);
        function getPopupClipCount : Integer;

        procedure setPinnedIndex(value : Integer);
        function getPinnedClip : string;
        function getPinnedClipCount : integer;

        function getInstallPath : string;

        procedure RunMacro(macro : string);

        function getCurrentWindow : integer;
        procedure setWindowHandle(value : Integer);
        procedure forceForegroundWindow(value : Integer);

        function getMouseX : Integer;
        function getMouseY : integer;
        procedure setMouseX(value : Integer);
        procedure setMouseY(value : Integer);

        procedure setWindowTitle(value : string);
        function findWindowByPartialTitle : integer;
        procedure setMaximized(value : Integer);
        procedure setMinimized(value : Integer);

        procedure setClipboardFormat(value : string);
        function getClipboardHasFormat : boolean;

        function getTitle : string;

        procedure writeDebugText(value : string);
        procedure showDebugText(value : string);

        procedure setClientName(value : string);
        procedure setClientX(value : integer);
        procedure setClientY(value : integer);
        function getClickInWindow : boolean;
        function getHandleFromClientname : integer;

        procedure setClientPath(value : string);
        function getFocusControl : boolean;
    protected

    public
    published
        // these are the properties that are exposed to JavaScript
        property Macro : string write RunMacro;
        property Clipboard : string read getClipboard write setClipboard;
        property PopupClip : string read getPopupClip;
        property PopupIndex : integer write setPopupIndex;
        property PopupCount : Integer read getPopupClipCount;

        property PinnedIndex : Integer write setPinnedIndex;
        property PinnedClip : string read getPinnedClip;
        property PinnedCount : Integer read getPinnedClipCount;

        property InstallPath : string read getInstallPath;

        property CurrentWindow : Integer read getCurrentWindow;
        property WindowHandle : integer write setWindowHandle;
        property ForegroundWindow : Integer write forceForegroundWindow;
        property MaximizeWindow : Integer write setMaximized;
        property MinimizeWindow : Integer write setMinimized;
        property MouseX : Integer read getMouseX write setMouseX;
        property MouseY : integer read getMouseY write setMouseY;

        property WindowTitle : string read getTitle;
        property SearchTerm : string write setWindowTitle;
        property FindWindowPartial : integer read findWindowByPartialTitle;
        property ClipFormat : string write setClipboardFormat;
        property HasFormat : Boolean read getClipboardHasFormat;

        property ShowDebug : string write showDebugText;
        property WriteDebug : string write writeDebugText;

        property clientName : string write setClientName;
        property clientX : integer write setClientX;
        property clientY : integer write setClientY;
        property clickInWindow : boolean read getClickInWindow;
        property classNameToHandle : integer read getHandleFromClientname;

        property clientPath : string write setClientPath;
        property focusControl : boolean read getFocusControl;
end;
{$M-}
function TScriptObject.getClipboard : string;
begin
    result := TClipboardGrabber.asText;
end;
procedure TScriptObject.setClipboard(value : string);
var duph : THANDLE;
    s : string;
begin

    s := value + #0;
    if TMyClipboard.OpenClipboard(Application.Handle,'SendText', 800) then begin
        Windows.EmptyClipboard;
        duph := UnitMisc.DupPointerToHandle(@s[1], length(s) * SizeOf(Char) );
        Windows.SetClipboardData(CF_UNICODETEXT, duph);

        TMyClipboard.CloseClipboard;
    end;

    clipbrd.Clipboard.SetTextBuf(@value[1]);
end;
procedure TScriptObject.RunMacro(macro : string);
begin
    Application.ProcessMessages;
    paste.SendMacro(macro);
    Application.ProcessMessages;

    // ISSUE - the first keystroke sent for the second
    // call of this command is ignored without ProcessMessages

    // ISSUE2 - the clipboard reported as "access denied" for the
    // second call of this command without the trailing ProcessMessages
end;
procedure TScriptObject.setPopupIndex(value : Integer);
begin
    fIndex := value;
end;
function TScriptObject.getPopupClip : string;
begin
    result := '';
    try
        result := ClipQueue.GetItemText(findex);
    except
    end;
end;
function TScriptObject.getPopupClipCount : Integer;
begin
    result := ClipQueue.GetQueueCount;
end;


procedure TScriptObject.setPinnedIndex(value : Integer);
begin
    fPinnedIndex := value;
end;
function TScriptObject.getPinnedClip;
begin
    result := '';
    try
        result := PinnedClipQueue.GetItemText(fPinnedIndex)
    except
    end;
end;
function TScriptObject.getPinnedClipCount : integer;
begin
    result := PinnedClipQueue.GetQueueCount;
end;

function TScriptObject.getInstallPath : string;
begin
    Result := unitMisc.GetAppPath;
end;

function TScriptObject.getCurrentWindow : integer;
begin
    result := GetForegroundWindow;
end;
procedure TScriptObject.setWindowHandle(value : Integer);
begin
    fHandle := value;
end;
procedure TScriptObject.forceForegroundWindow(value : Integer);
begin
    TFocusManager.ForceForeground(value);
end;


procedure TScriptObject.setMaximized(value : Integer);
begin
    Windows.ShowWindow(value, SW_MAXIMIZE);
end;
procedure TScriptObject.setMinimized(value : Integer);
begin
    Windows.ShowWindow(value, SW_MINIMIZE);
end;

function TScriptObject.getMouseX : Integer;
begin
    result := Mouse.CursorPos.X;
end;
function TScriptObject.getMouseY : integer;
begin
    result := Mouse.CursorPos.Y;
end;


procedure TScriptObject.setMouseX(value : integer);
begin
    SetCursorPos(value, Mouse.CursorPos.Y);
end;
procedure TScriptObject.setMouseY(value : Integer);
begin
    SetCursorPos(Mouse.CursorPos.X, value);
end;

// clickInWindow("ClassName", X, Y)
procedure TScriptObject.setClientName(value : string);
begin
    fclientName := value;
end;
procedure TScriptObject.setClientX(value : integer);
begin
    fclientX := value;
end;
procedure TScriptObject.setClientY(value : integer);
begin
    fclientY := value;
end;
function TScriptObject.getClickInWindow : boolean;
var h : THandle;
    pt : TPoint;
    startingPT : TPoint;
begin
    result := false;
    h := self.getHandleFromClientname;
    if h = 0 then begin
        self.writeDebugText('clickInWindow() - clientName window not found.');
        EXIT;
    end;
    TFocusManager.ForceForeground(h);

    startingPT := Mouse.cursorPos;
    pt.X := fclientX;
    pt.Y := fclientY;
    ClientToScreen(h, pt);

    SetCursorPos(pt.x,pt.y);
    mouse_event(MOUSEEVENTF_LEFTDOWN,0,0,0,0);
    Sleep(100);
    mouse_event(MOUSEEVENTF_LEFTUP,0,0,0,0);


    SetCursorPos(startingPT.X, startingPT.Y);
    result := true;
end;
function TScriptObject.getHandleFromClientname : integer;
begin
    result := FindWindowEx(0,0,PWideChar(fclientName),nil);
end;

// focusControlByPath("Path")
procedure TScriptObject.setClientPath(value : string);
begin
    fclientPath := value;
end;
function TScriptObject.getFocusControl : boolean;
begin
    result := TUIAutomation.FocusControl(fclientPath);
    if not result then begin
        self.writeDebugText('focusControl() - control not found.');
    end;
end;


var
    enumTitle : string;
    enumHandle : HWND;

function getWindowTitle(hwnd: HWND) : string;
var sz : integer;
begin
    sz := GetWindowTextLength(hwnd);
    if (sz = 0) then EXIT;
    inc(sz, 1); // don't ask

    SetLength(result, sz);
    GetWindowText(hwnd, PChar(result), sz);
end;
function EnumWindowsProc(hwnd: HWND; lParam: LPARAM): BOOL; stdcall;
var
    s, strfind : string;
    sz : integer;
begin
    result := True;

    if not IsWindowVisible(hwnd) then Exit;
    if (GetWindow(hwnd, GW_OWNER)<>0) then EXIT;
    if ( GetWindowLongPtr(hwnd, GWL_STYLE) and WS_EX_APPWINDOW)=0 then Exit;

    s := getWindowTitle(hwnd);
    s := UpperCase(s);

    if Pos(enumTitle, s) <> 0 then begin
        enumHandle := hwnd;
        result := false;
    end;
end;
procedure TScriptObject.setWindowTitle(value : string);
begin
    fTitle := value;
end;
function TScriptObject.findWindowByPartialTitle : Integer;
begin
    enumTitle := UpperCase(ftitle);
    Result := 0;
    enumHandle := 0;
    Windows.SetLastError(0);
    if EnumWindows(@EnumWindowsProc, 0)  then begin
        // Delphi or Windows is dumb
        // TRUE means not found and FALSE means found
        result := enumHandle;
    end else begin
        if not (GetLastError = 0) then begin
            FrmDebug.AppendLog('EnumWindowsFailed', true);
        end else begin
            result := enumHandle;
        end;
    end;
end;
function TScriptObject.getTitle : string;
begin
    result := '';
    if fhandle = 0 then EXIT;

    result := getWindowTitle(fHandle);
end;



procedure TScriptObject.setClipboardFormat(value : string);
begin
    fFormat := value;
end;
function TScriptObject.getClipboardHasFormat : boolean;
begin
    result := false;
    if fFormat = 'RICH' then begin
        result := Windows.IsClipboardFormatAvailable(GetCF_RICHTEXT);
    end else if fFormat = 'HTML' then begin
        result := Windows.IsClipboardFormatAvailable(GetCF_HTML);
    end else if fFormat = 'TEXT' then begin
        result := Windows.IsClipboardFormatAvailable(CF_UNICODETEXT);
    end else if fFormat = 'PIC' then begin
        result := Windows.IsClipboardFormatAvailable(CF_DIB);
    end else if fFormat = 'FILES' then begin
        result := Windows.IsClipboardFormatAvailable(CF_HDROP);
    end;

end;


procedure TScriptObject.writeDebugText(value : string);
begin
    FrmDebug.AppendLog('JavaScript(DEBUG): ' + value );
end;
procedure TScriptObject.showDebugText(value : string);
begin
    ShowMessage(value);
end;



var ScriptObject : TScriptObject;



constructor TJScript.Create();
begin
    inherited create(Application);

    ScriptObject := TScriptObject.Create(Application);

    CoInitializeEx(nil, COINIT_APARTMENTTHREADED);
    OleCheck(CoCreateInstance(CLSID_JScript, nil, CLSCTX_INPROC_SERVER, IID_IActiveScript, FScript));
    OleCheck(FScript.SetScriptSite(Self as IActiveScriptSite));
    OleCheck(FScript.AddNamedItem(ArsClipObject, SCRIPTITEM_ISVISIBLE or SCRIPTITEM_ISSOURCE));

    if Supports(FScript, IActiveScriptParse, FParse) then OleCheck(FParse.InitNew);
end;

const
JS_NL = #13#10;
NL = JS_NL;

JS_MAIN_FUNCTION = 'function main(clipboardStr)';
JS_MAIN_HEADER =
    JS_MAIN_FUNCTION + ' {' + JS_NL +
    '// see [View > JavaScript Reference] for special functions for ArsClip' + JS_NL ;
JS_MAIN_FOOTER =
    '}';
JS_SCRIPTNAME = 'main.js';
JS_SCRIPT_FOOTER =
    NL +
    'function manuallyExecuteMacro(macro) {_ArsClip.Macro = macro;} ' + NL +
    'function saveToPopupClips(clip) {_ArsClip.Macro = ''[newclip=\"''+clip.replace(/\"/g,''""'')+''\"]'';}' +NL +
    'function getCurrentClipboard() { return _ArsClip.Clipboard;}' +NL+
    'function setClipboard(clip) { _ArsClip.Clipboard = clip; }'+NL+

    'function getPopupClip(index) { _ArsClip.PopupIndex = index; return _ArsClip.PopupClip;}'+NL+
    'function getPopupClipCount() { return _ArsClip.PopupCount; }'+NL+

    'function getPinnedClip(index) { _ArsClip.PinnedIndex = index; return _ArsClip.PinnedClip;}'+NL+
    'function getPinnedClipCount() { return _ArsClip.PinnedCount; }'+NL+

    'function getInstallPath() { return _ArsClip.InstallPath; }'+NL+

    'function getCurrentWindow() { return _ArsClip.CurrentWindow; }'+NL+
    'function findWindowByPartialTitle(title) { _ArsClip.SearchTerm = title; return _ArsClip.FindWindowPartial; }'+NL+

    'function setForegroundWindow(index) { _ArsClip.ForegroundWindow = index; }'+NL+
    'function setWindowMaximized(index) { _ArsClip.MaximizeWindow = index; }'+NL+
    'function setWindowMinimized(index) { _ArsClip.MinimizeWindow = index; }'+NL+
    'function getWindowTitle(index) { _ArsClip.WindowHandle = index; return _ArsClip.WindowTitle; }'+NL+

    'function getMouseX() { return _ArsClip.MouseX; }'+NL+
    'function getMouseY() { return _ArsClip.MouseY; }'+NL+
    'function setMousePos(x, y) { _ArsClip.MouseX = x; _ArsClip.MouseY = y; }'+NL+
    'function clipboardHasFormat(clipformat) { _ArsClip.ClipFormat = clipformat; return _ArsClip.HasFormat;}'+NL+

    'function showDebugText(text) { _ArsClip.showDebug = text; }'+NL+
    'function writeDebugText(text) { _ArsClip.writeDebug = text; }'+NL+

    'function clickInWindow(classname,x,y) {'+
        '_ArsClip.clientX = x;'+
        '_ArsClip.clientY = y;'+
        '_ArsClip.clientName = classname;'+
        'var b = _ArsClip.clickInWindow; } '+NL+

    'function focusControlByPath(path) {'+
        '_ArsClip.clientPath = path;'+
        'var b = _ArsClip.focusControl; }'+NL+

    'main(_ArsClip.Clipboard);';
const
    JS_START_MACRO_VAR = 'start_ac_macro:';
    JS_END_MACRO_VAR = 'end_ac_macro:';
    JS_START_MACRO = JS_START_MACRO_VAR+#13#10;
    JS_END_MACRO = JS_END_MACRO_VAR+#13#10;


class function TJScript.isJavaScript(str : string) : boolean;
var i : integer;
begin
    i := ansiPos(JS_MAIN_FUNCTION, str);
    result := (i > 0);
end;

class function TJScript.getStartMacro : string;
begin
    result := JS_MAIN_HEADER + JS_START_MACRO;
end;
class function TJScript.getEndMacro : string;
begin
    result := JS_END_MACRO + JS_MAIN_FOOTER;
end;
class function TJScript.getStartNoMacro : string;
begin
    result := JS_MAIN_HEADER;
end;
class function TJScript.getEndNoMacro : string;
begin
    result := JS_MAIN_FOOTER;
end;

function TJScript.getLastErr : string;
begin
    result := fLastErr;
end;



function TJScript.executeJS(paste : TPaste; javascript : string) : boolean;
    function encodeACMacros(javascript : string) : string;
    var
        s : string;
        macro : string;
        function createJSLitteralString(const str : string) : string;
        begin
            result := str;
            if length(result) = 0 then begin
                result := '""';
                EXIT;
            end;

            if result[length(result)] = #0 then begin
                delete(result,length(result),1);
            end;
            result := ReplaceText(result, '\', '\\');
            result := ReplaceText(result, '''', '\''');
            result := ReplaceText(result, '"', '\"');
            result := ReplaceText(result, #13, '\r');
            result := ReplaceText(result, #10, '\n');
            result := '"' + result + '"';
        end;
        function MacroToJS(const macro : string) : string;
        var
            s : string;
        begin
            result := '';
            if length(macro) = 0 then begin
                Exit;
            end;
            s := macro;
            if macro[length(macro)] = #10 then begin
                delete(s,length(s)-1, 2);
            end;
            s := createJSLitteralString(s);
            result := 'manuallyExecuteMacro('+ s +');' + NL;
        end;
    begin
        while (javascript <> '') do begin
            result := result + TokenString(javascript,JS_START_MACRO_VAR,false);
            if (javascript <> '') then begin
                TokenString(javascript, JS_NL,false);
                macro := TokenString(javascript,JS_END_MACRO_VAR,false);
                TokenString(javascript, JS_NL,false);
                result := result + macroToJS(macro);
            end;
        end;
    end;
begin
    FrmDebug.AppendLog('executeJS started');

    terminated := false;
    fLastErr := '';
    ScriptObject.fPaste := paste;
    javascript := encodeACMacros(javascript);
    javascript := javascript + JS_SCRIPT_FOOTER;

    RunScript(javascript);
    while not terminated do begin
        Sleep(50);
    end;

    FrmDebug.AppendLog('executeJS terminated');
end;
procedure TJScript.RunScript(script : string);
var
    info: EXCEPINFO;
begin
    try
        OleCheck(FParse.ParseScriptText(PWideChar(script), nil, nil, nil, 0, 0,
            SCRIPTITEM_ISVISIBLE or SCRIPTITEM_ISPERSISTENT, nil, Info));
        OleCheck(FScript.SetScriptState(SCRIPTSTATE_CONNECTED));
        OleCheck(FScript.SetScriptState(SCRIPTSTATE_DISCONNECTED));
    except on e : Exception do
        begin
            terminated := true;
            if (fLastErr = '') then begin
                flastErr := e.Message;
            end;
        end;
    end;
end;




//
// IActiveScriptSite interface stuff
//
function TJScript.GetDocVersionString(out pbstrVersion: WideString): HResult;
begin
    pbstrVersion := '1.0';
    Result := S_OK;
end;
function TJScript.GetItemInfo(pstrName: PWideChar; dwReturnMask: LongWord; out ppiunkItem, ppti: IInterface): HResult;
begin
    result := S_FALSE;

    if SameText(ArsClipObject, pstrName) then begin
        if dwReturnMask and SCRIPTINFO_IUNKNOWN <> 0 then begin
            ppiunkItem := TObjectDispatch.Create(ScriptObject) as IInterface;
            result := S_OK;
            EXIT;
        end;

        if dwReturnMask and SCRIPTINFO_ITYPEINFO <> 0 then begin
            Result := TYPE_E_ELEMENTNOTFOUND;
            EXIT;
        end;
    end;
end;
function TJScript.GetLCID(out plcid: LongWord): HResult;
begin
    plcid := Windows.GetThreadLocale;
    result := S_OK;
end;
function TJScript.OnEnterScript: HResult;
begin
    result := S_OK;
end;
function TJScript.OnLeaveScript: HResult;
begin
    result := S_OK;
end;
function TJScript.OnScriptError(const pscripterror: IActiveScriptError): HResult;
var
    Info: EXCEPINFO;
    scontext: cardinal;
begin
    terminated := true;
    if ActiveX.Succeeded(pscripterror.GetExceptionInfo(Info)) and
        ActiveX.Succeeded(pscripterror.GetSourcePosition(scontext, flastLine, fLastPos)) then  begin
        fLastErr := Info.bstrDescription;
    end;
    Result := SCRIPT_E_REPORTED;
end;
function TJScript.OnScriptTerminate(var pvarResult: OleVariant; var pexcepinfo: EXCEPINFO): HResult;
begin
    result := S_OK;
    terminated := true;
end;
function TJScript.OnStateChange(ssScriptState: tagSCRIPTSTATE): HResult;
begin
    result := S_OK;
    if ssScriptState = SCRIPTSTATE_DISCONNECTED then begin
        terminated := True;
    end;
end;


initialization
finalization
begin
    TFrmDebug.MeOnlyAppend('UnitJScript', false);
end;

end.
