unit UnitUIAutomation;

interface

uses Winapi.Windows, Winapi.Messages, System.Types,
    UIAutomationClient_TLB,    System.Variants, ComObj,
    UnitTextInspectorData, Vcl.StdCtrls, Generics.collections;



Type TAutomationList = TList<string>;
type TUIAutomation = class(TOBject)
    private
        class function TryAndGet(
            element : IUIAutomationElement;
            patternID : TControlPatternID;
            var pattern : IUnknown
        ) : boolean;
        class function GetProperty(element : IUIAutomationElement; id : TPropertyID) : OleVariant; overload;
        class function GetProperty(cache : IUIAutomationCacheRequest;element : IUIAutomationElement; id : TPropertyID) : OleVariant; overload;
        class function GetFullPathName(element :IUIAutomationElement) : string; overload;
        class function FullPathNameToElement(path : string; var element : IUIAutomationElement) : boolean;
        class function GetAutomation : IUIAutomation;
        class function isContainer(uiaType : UIA_ControlType) : boolean;
        class procedure FillInfo(C : TTICInfo; element : IUIAutomationElement);
        class function GetGridRowText(element : IUIAutomationElement; var s : string) : boolean;
        class function GetHeaderText(element: IUIAutomationElement; var s : string) : boolean;
        class function GetListItemText(element : IUIAutomationElement; pt : Tpoint; var s : string) : boolean;
        class function GetCustomText(element : IUIAutomationElement;pt : TPoint; var subrect : Trect; var s : string) : boolean;
        class function FindChildByPoint(element :IUIAutomationElement;pt : TPoint; var child : IUIAutomationElement ) : boolean; overload;
        class function FindChildByPoint(element :IUIAutomationElement;pt : TPoint; var child : IUIAutomationElement; var idx : integer ) : boolean; overload;
        class function FindChildByPoint(element :IUIAutomationElement; cType :UIA_ControlType; pt : TPoint; var child : IUIAutomationElement; var idx : integer ) : boolean; overload;
        class function FindChildByIndex(element : IUIAutomationElement; cType :UIA_ControlType; index : integer; var child : IUIAutomationElement) : boolean;
        class function GetTextOrLink(element : IUIAutomationElement; var s : string ) : boolean;
        class function GetTextControlText(element : IUIAutomationElement; var s : string) : boolean;
    public
        class function FindControl(c : TTICInfo) : boolean;
        class function GetAssociatedListView(header : TTICInfo; var listview : TTICInfo) : boolean;
        class function GetColumnHeaders(c : TTIcInfo; var s : string) : boolean;
        class function GetColumnIndex(c : TTICInfo; var idx : integer) : boolean;
        class function GetListViewTextAsGrid(c : TTICInfo; row, col : integer; var s : string) : boolean;
        class function GetListViewRow(c : TTICinfo; var row : integer) : boolean; overload;
        class function GetListViewRow(c : TTICinfo; var row : integer; var listItem : IUIAutomationElement) : boolean; overload;
        class function GetListViewRowCount(c : TTICInfo; var count : integer) : boolean;
        class function GetListViewSelectedCount(c : TTICINfo; var count : integer) : boolean;
        class function GetListViewSelectedRows(c : TTICINfo; rows : TList<TAutomationList>) : boolean; overload;
        class function GetListViewSelectedRows(c : TTICINfo; col : integer; rows : TList<TAutomationList>) : boolean; overload;
        class function GetTreeViewPathText(treenode : TTICInfo; var s : string) : boolean;
        class function GetComboBoxList(combo : TTICInfo; var s : string) : boolean;
        class function GetChildrenTextFromContainer(c : TTICInfo; textList : TList<string>) : boolean;

        class procedure InspectControl(c : TTICInfo );

        class function GetFullPathName(h : THandle) : string; overload;
        class function FocusControl(path : string) : boolean;
        class function GetIdProperty(h : THandle) : string;
        class function HandleFromPoint(pt : TPoint) : THandle;
        class function GetParent(c : TTICInfo; Content : boolean = true) : TTICinfo;

        class function isActuallyHeaderItem(c : TTICInfo) : boolean;
end;

implementation

uses SysUtils, UnitToken, UnitMisc, UnitFrmDebug;



class function TUIAutomation.GetAutomation : IUIAutomation;
begin
    try
        result := CoCUIAutomation.Create;
    except
        result := nil;
    end;
end;
class function TUIAutomation.GetProperty(element : IUIAutomationElement; id : TPropertyID) : OleVariant;
begin
    Element.GetCurrentPropertyValue(Integer(id), result);
end;
class function TUIAutomation.GetProperty(cache : IUIAutomationCacheRequest;element : IUIAutomationElement; id : TPropertyID) : OleVariant;
begin
    element.GetCachedPropertyValue(Integer(id), result);
end;
// NOTES: This is incredible slow to try to find a unique name for a control
// it has to create a hierarchy from the top level window down the the control

class function TUIAutomation.GetFullPathName(element :IUIAutomationElement) : string;
var
    Automation : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    root, child, childX, tempchild, parent : IUIAutomationElement;
    str : string;
    same : longbool;
    classname, controlName, controlDesc,
    childClassname, childControlName, childDesc : string;
    tempChildH, childH : THandle;
    classCnt, nameCnt, descCnt : integer;
    fail : boolean;
    procedure appendResult(s : string);
    begin
        if result <> '' then result := ' ' + result;
        result := '('+s+')' + result;
    end;
begin
    Automation := GetAutomation;
    if not Assigned(automation) then EXIT;

    Automation.Get_ContentViewWalker(tw);
    if not assigned(tw) then EXIT;



    Automation.GetRootElement(root);
    child := element;
    fail := false;
    repeat
        className := GetProperty(child, UIA_ClassNamePropertyId);
        controlName := GetProperty(child, UIA_NamePropertyId);
        controlDesc := GetProperty(child, UIA_FullDescriptionPropertyId);

        // test for UNIQUE id among siblings
        str := GetProperty(child, UIA_AutomationIdPropertyId);
        if str <> '' then begin
            appendX(result,'[ID:'+str+']',' ', true);
        end;

        tw.GetParentElement(child, parent);
        Automation.CompareElements(parent, root, integer(same));
        if (longbool(same)) then begin
            if (str='') then begin
                str := GetProperty(child, UIA_ClassNamePropertyId);
                appendX(result, '[CLASS(0):' +str+ ']', ' ', true);
            end;
            parent := nil;
            CONTINUE;
        end;

        // test for alternatives
        if (str='') and (classname <> '') then begin
            tw.GetFirstChildElement(Parent, ChildX);
            while (child <> nil) do begin
                childClassname := GetProperty(ChildX, UIA_ClassNamePropertyId);
                if sametext(classname, childClassname) then begin
                    inc(classCnt);
                    Automation.CompareElements(child, childX, integer(same));
                    if same then BREAK;
                end;
                tw.GetNextSiblingElement(childX, tempChild);
                childX := tempChild;
            end;
            appendResult('CLASS('+IntToStr(classCnt)+'):'+classname);
        end;
        child := parent;
    until (parent = nil);
end;
class function TUIAutomation.FullPathNameToElement(path : string; var element :IUIAutomationElement) : boolean;
var
    Automation : IUIAutomation;
    child, parent : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    Condition : IUIAutomationCondition;
    controlName, controlType, controlID, controlIdx : string;
    idx, i,j, cnt : integer;

    function GetIdPropertyCondition(controlName : string) :IUIAutomationCondition;
    var ovar : OleVariant;
    begin
        ovar := controlName;
        Automation.CreatePropertyConditionEx(
            Integer(UIA_AutomationIdPropertyId),
            ovar,
            PropertyConditionFlags_IgnoreCase,
            result
        );
    end;
    function GetClassNameCondition(controlName : string) :IUIAutomationCondition;
    var ovar : OleVariant;
    begin
        ovar := controlName;
        Automation.CreatePropertyConditionEx(
            Integer(UIA_ClassNamePropertyId),
            ovar,
            PropertyConditionFlags_IgnoreCase,
            result
        );
    end;
begin
    result := false;
    Automation := GetAutomation;
    if not Assigned(automation) then EXIT;

    Automation.GetRootElement(parent);
    repeat
        UnitToken.TokenString(path,'[', true);
        controlType := UnitToken.TokenString(path, ':',false);
        idx := 0;
        if pos('(', controlType) > 0 then begin
            controlID := UnitToken.TokenString(controlType, '(', false);
            controlIdx := UnitToken.TokenString(controlType, ')', false);
            idx := StrToInt(controlIdx);
        end else begin
            controlID := controlType;
        end;
        controlName := UnitTOken.TokenString(path, ']');
        if SameText(controlId,'ID') then begin
            condition := GetIdPropertyCondition(controlName);
        end else if SameText(controlId, 'CLASS') then begin
            condition := GetClassNameCondition(controlName);
        end;

        if idx = 0 then begin
            parent.FindFirst(
                TreeScope_Children,
                condition,
                child
            );

            parent := child;
        end else begin
            parent.FindAll(
                TreeScope_Children,
                condition,
                elements
            );
            if elements = nil then begin
                parent := nil;
                BREAK;
            end;
            elements.Get_Length(cnt);
            if idx < cnt then begin
                elements.GetElement(idx, parent);
            end else begin
                parent := nil;
            end;
        end;

    until (path = '') or (parent=nil);



    result :=  parent <> nil;
    if result then begin
        element := parent;
    end;
end;
class function TUIAutomation.TryAndGet(
    element : IUIAutomationElement;
    patternID : TControlPatternID;
    var pattern : IUnknown
) : boolean;
var iInter : IInterface;
    g : TGUID;
begin
    result := false;
    try
        case patternId of
        UIA_ValuePatternID: g := IID_IUIAutomationValuePattern;
        UIA_TextPatternId: g := IID_IUIAutomationTextPattern;
        UIA_GridPatternId: g := IID_IUIAutomationGridPattern;
        UIA_GridItemPatternId: g := IID_IUIAutomationGridItemPattern;
        UIA_MultipleViewPatternId: g := IID_IUIAutomationMultipleViewPattern;
        UIA_SelectionPatternId: g := IID_IUIAutomationSelectionPattern;
        UIA_SelectionItemPatternId: g := IID_IUIAutomationSelectionItemPattern;
        end;
        if not (element.GetCachedPattern(Integer(patternID), iInter) = S_OK) then begin
            OleCheck(element.GetCurrentPattern(Integer(patternID), iInter));
        end;
        if assigned(iInter) then begin
            OleCheck(iInter.QueryInterface(g, pattern));
            result := pattern <> nil;
        end;
    except on e : Exception do
        begin

        end;

    end;
end;


class function TUIAutomation.FindChildByIndex(
    element : IUIAutomationElement; cType :UIA_ControlType;
    index : integer; var child : IUIAutomationElement) : boolean;
var a : IUIAutomation;
    cache : IUIAutomationCacheRequest;
    condition : IUIAutomationCondition;
    elements : IUIAutomationElementArray;
    uiaType : UIA_ControlType;
    i, j : integer;
    r : TRect;
    tempchild : IUIAutomationElement;
    same : longbool;
begin
    result := false;
    a := GetAutomation;
    a.CreateCacheRequest(cache);
    if (cType= UIA_Unknown) then begin
        a.CreatePropertyCondition(
            Integer(UIA_IsContentElementPropertyId),
            true,
            condition
        )
    end else begin
         a.CreatePropertyCondition(
            Integer(UIA_ControlTypePropertyId),
            cType,
            condition
        )
    end;
    cache.Set_TreeScope(TreeScope_element);

    element.FindAllBuildCache(
       TreeScope_Children, condition, cache, elements
    );

    elements.Get_Length(j);
    for i := 0 to j-1 do begin
        elements.GetElement(i, tempchild);
        a.CompareElements(element, tempchild, Integer(same));
        if same then begin
            result := true;
            child := tempchild;
        end;
    end;

end;
class function TUIAutomation.FindChildByPoint(
    element :IUIAutomationElement; cType : UIA_ControlType; pt : TPoint;
    var child : IUIAutomationElement; var idx : integer ) : boolean;
var a : IUIAutomation;
    cache : IUIAutomationCacheRequest;
    condition : IUIAutomationCondition;
    elements : IUIAutomationElementArray;
    uiaType : UIA_ControlType;
    ws : WideString;
    i, j : integer;
    r : TRect;
begin
    result := false;
    a := GetAutomation;
    a.CreateCacheRequest(cache);
    //a.CreateTrueCondition(condition);
    if (cType= UIA_Unknown) then begin
        a.CreatePropertyCondition(
            Integer(UIA_IsContentElementPropertyId),
            true,
            condition
        )
    end else begin
         a.CreatePropertyCondition(
            Integer(UIA_ControlTypePropertyId),
            cType,
            condition
        )
    end;


    cache.AddProperty(integer(UIA_BoundingRectanglePropertyId));
    cache.AddProperty(integer(UIA_ControlTypePropertyId));
    cache.Set_TreeScope(TreeScope_element);

    element.FindAllBuildCache(
       TreeScope_Children, condition, cache, elements
    );

    elements.Get_Length(j);
    for i := 0 to j-1 do begin
        elements.GetElement(i, child);
        uiaType := GetProperty(cache, child, UIA_ControlTypePropertyId);
        if child.Get_CachedBoundingRectangle(tagRect(r)) = S_OK then
            if ptinrect(r, pt) then begin
                result := true;
                idx := i;
                EXIT;
            end;
    end;
end;
class function TUIAutomation.FindChildByPoint(
    element :IUIAutomationElement;pt : TPoint;
    var child : IUIAutomationElement; var idx : integer ) : boolean;
begin
    TUIAutomation.FindChildByPoint(
        element, UIA_UNKNOWN,pt,
    child, idx);
end;
class function TUIAutomation.FindChildByPoint(
    element :IUIAutomationElement;pt : TPoint;
    var child : IUIAutomationElement ) : boolean;
var i : integer;
begin
    TUIAutomation.FindChildByPoint(element, pt, child, i)
end;



class function TUIAutomation.GetTextOrLink(element : IUIAutomationElement; var s : string ) : boolean;
var
    uiaType : UIA_ControlType;
    str, valueProperty : string;
begin
    uiaType := GetProperty(element, UIA_ControlTypePropertyId);
    str := GetProperty(element, UIA_NamePropertyId);
    if uiaType = UIA_HyperlinkControlTypeId then begin
        valueProperty := GetProperty(element, UIA_ValueValuePropertyId);
        if str <> '' then str := str + #13#10;
        str := str + valueProperty;
    end else if uiaType = UIA_EditControlTypeId then begin
        str := GetProperty(element, UIA_ValueValuePropertyId);
    end;

    result := str <> '';

    if result then s := str;
end;

class function TUIAutomation.GetGridRowText(element : IUIAutomationElement; var s : string) : boolean;
var
    GridElement, tempGridElement : IUIAutomationElement;
    GridItemPattern : IUIAutomationGridItemPattern;
    GridPattern : IUIAutomationGridPattern;
    ValPattern : IUIAutomationValuePattern;
    ViewPattern : IUIAutomationMultipleViewPattern;
    ViewName : string;
    i, j, k : integer;
    ws : WideString;
    a : IUIAutomation;
    condition : IUIAutomationCondition;
    c : IUIAutomationCacheRequest;
    tempElement : IUIAutomationElement;
    r : TRect;
const VIEW_DETAILS = 5;
begin
    s := '';
    a := GetAutomation;

    result := GetProperty(element, UIA_IsGridItemPatternAvailablePropertyId);


    if not result then EXIT;
    if not TUIAutomation.TryAndGet(
        element,
        UIA_GridItemPatternId,
        IUnknown(GridItemPattern)
    ) then EXIT;

    GridItemPattern.Get_CurrentContainingGrid(gridElement);

    result := result and GetProperty(gridElement, UIA_IsMultipleViewPatternAvailablePropertyId);
    if not result then EXIT;

    if not TUIAutomation.TryAndGet(
        GridElement,
        UIA_MultipleViewPatternId,
        IUnknown(ViewPattern)
    ) then EXIT;

    ViewPattern.Get_CurrentCurrentView(i);
    ViewPattern.GetViewName(i, ws);
    ViewName := ws;
    if i <> VIEW_DETAILS then EXIT;


    GridItemPattern.Get_CurrentRow(j);
    if not TUIAutomation.TryAndGet(
        gridElement,
        UIA_GridPatternId,
        IUnknown(GridPattern)
    ) then EXIT;


    GridPattern.Get_CurrentColumnCount(i);
    for k := 0 to i-1 do begin

        GridPattern.GetItem(j, k, tempElement);
        if TUIAutomation.TryAndGet(
            tempElement,
            UIA_ValuePatternId,
            IUnknown(ValPattern)
        ) then begin
            ValPattern.Get_CurrentValue(ws);
            if ws <> '' then begin
                if s <> '' then s := s + ' ';
                s := s + ws;
            end;
        end;
    end;
end;
class function TUIAutomation.GetHeaderText(element: IUIAutomationElement; var s : string) : boolean;
var
    uiaType : UIA_ControlType;
    ws : WideString;
    tw : IUIAutomationTreeWalker;
    a : IUIAutomation;
    parent, child : IUIAutomationElement;
    condition : IUIAutomationCondition;
    elements : IUIAutomationElementArray;
    nameProperty : string;
    i, j : integer;
begin
    result := false;
    uiaType := GetProperty(element, UIA_ControlTypePropertyId);
    case uiaType of
    UIA_SplitButtonControlTypeId:
        begin
            // Explorer uses split buttons as column headers
            // but don't identify themselves as column headers
            a := GetAutomation;
            a.Get_ControlViewWalker(tw);

            tw.GetParentElement(element, parent);
            uiaType := GetProperty(parent, UIA_ControlTypePropertyId);
            if not (uiaType = UIA_HeaderControlTypeId) then EXIT;

            s := GetPRoperty(element, UIA_NamePropertyId);
            result := true;
        end;
    UIA_HeaderItemControlTypeId:
        begin
            s := GetPRoperty(element, UIA_NamePropertyId);
            result := true;
        end;
    UIA_HeaderControlTypeId:
        begin
            a := GetAutomation;
            a.CreateTrueCondition(condition);
            element.FindAll(
                TreeScope_Children,
                condition,
                elements
            );

            elements.Get_Length(j);
            for i := 0 to j-1 do begin
                elements.GetElement(i, child);
                nameProperty := GetPRoperty(element, UIA_NamePropertyId);
                if nameProperty <> '' then begin
                    if s <> '' then s := s + ' ';
                    s := s + nameProperty;
                end;
            end;
        end;
    else
        begin

        end;
    end;
end;
class function TUIAutomation.GetListItemText(element : IUIAutomationElement; pt : TPoint; var s : string) : boolean;
var
    child, parent, lastFound : IUIAutomationElement;
    str : string;
begin
    result := false;
    str := GetProperty(element, UIA_NamePropertyId);
    if str <> '' then begin
        s := str;
        result := true;
        EXIT;
    end;


    parent := element;
    repeat
        if FindChildByPoint(parent, pt, Child) then begin
            lastFound := child;
        end else begin
            child := nil;
        end;
        parent := child;
    until child = nil;

    if lastFound <> nil then begin
        GetTextOrLink(lastFound, str);
    end;
    result := lastFound <> nil;
    if result then s := str;
end;
class function TUIAutomation.GetCustomText(element : IUIAutomationElement; pt : TPoint; var subrect : Trect; var s : string) : boolean;
var a : IUIAutomation;
    lastFound, parent, child : IUIAutomationElement;
    r : TRect;
begin
    result := false;

    parent := element;
    repeat
        if FindChildByPoint(parent, pt, Child) then begin
            lastFound := child;
        end else begin
            child := nil;
        end;
        parent := child;
    until child = nil;

    if lastFound <> nil then begin
        GetTextOrLink(lastFound, s);
        r := trect.Empty;
        if lastFound.Get_CurrentBoundingRectangle(tagrect(r)) = S_OK then begin
            if r.Width <> 0 then begin
                subrect := r;
            end;
        end;
    end;
    result := lastFound <> nil;
end;
class function TUIAutomation.GetTextControlText(element : IUIAutomationElement; var s : string) : boolean;
var
    str : string;
    TextRange : IUIAutomationTextRange;
    TextPattern : IUIAutomationTextPattern;
    ws : WideString;
begin
    result := false;
    str := GetProperty(element, UIA_NamePropertyId);
    if str = '' then begin
        str := GetProperty(element, UIA_ValueValuePropertyId);
    end else begin
        result := true;
        s := str;
    end;

    if not result then begin
        // TODO: support text ranges for text that cannot
        // be focused
        if not TryAndGet(element, UIA_TextPatternId, IUnknown(TextPattern)) then EXIT;
        if not Assigned(TextPattern) then EXIT;

        TextPattern.Get_DocumentRange(textRange);
        TextRange.GetText(-1, ws);
        if ws <> '' then begin
            result := true;
            s := ws;
        end;
    end;
end;
class function TUIAutomation.isContainer(uiaType : UIA_ControlType) : boolean;
    begin
        result :=
        (uiaType = UIA_DocumentControlTypeId) or
        (uiaType = UIA_PaneControlTypeId) or
        (uiaType = UIA_GroupControlTypeId)or
        // Tab Controls on a web page return the entire document
        //(uiaType = UIA_TabControlTypeId) or

        (uiaType = UIA_ListControlTypeId) or
       // (uiaType = UIA_ListItemControlTypeId) or // it's complicated
        (uiaType = UIA_TableControlTypeId) or
        (uiaType = UIA_CustomControlTypeId) or

        (uiaType = UIA_DataGridControlTypeId) or
        (uiatype = UIA_DataItemControlTypeId);
    end;
//
// Public Interface
//
class function TUIAutomation.isActuallyHeaderItem(c : TTICInfo) : boolean;
var
    uiaType : UIA_ControlType;
    tw : IUIAutomationTreeWalker;
    a : IUIAutomation;
    element, parent, child : IUIAutomationElement;
    condition : IUIAutomationCondition;

    i, j : integer;
begin
    result := false;
    element := c.UIA.Element;
    uiaType := GetProperty(  element, UIA_ControlTypePropertyId);
    case uiaType of
    UIA_SplitButtonControlTypeId:
        begin
            // Explorer uses split buttons as column headers
            // but don't identify themselves as column headers
            a := GetAutomation;
            a.Get_ControlViewWalker(tw);

            tw.GetParentElement(element, parent);
            uiaType := GetProperty(parent, UIA_ControlTypePropertyId);
            if not (uiaType = UIA_HeaderControlTypeId) then EXIT;
            result := true;
        end;
    UIA_HeaderItemControlTypeId:
        begin
            result := true;
        end;
    end;
end;
class function TUIAutomation.GetColumnHeaders(c : TTIcInfo; var s : string) : boolean;
var
    uiaType : UIA_ControlType;
    tw : IUIAutomationTreeWalker;
    a : IUIAutomation;
    element, parent, tempParent, child : IUIAutomationElement;
    condition : IUIAutomationCondition;
    elements : IUIAutomationElementArray;
    i, j, k : integer;
    ws : WideString;
    str : string;
begin
    result := false;
    element := c.UIA.Element;
    uiaType := GetProperty(  element, UIA_ControlTypePropertyId);
    a := GetAutomation;
    a.Get_ControlViewWalker(tw);

    case uiaType of
    UIA_SplitButtonControlTypeId:
        begin
            // Explorer uses split buttons as column headers
            // but don't identify themselves as column headers
            tw.GetParentElement(element, tempParent);
            parent := tempParent;
            uiaType := GetProperty(parent, UIA_ControlTypePropertyId);
            if uiaType <> UIA_HeaderControlTypeId then begin
                tw.GetParentElement(parent, tempParent);
                parent := tempParent;
                uiaType := GetProperty(  parent, UIA_ControlTypePropertyId);
            end;
        end;
    UIA_HeaderItemControlTypeId:
        begin
            tw.GetParentElement(element, parent);
        end;
    end;

    if parent <> nil then begin
        a.CreateTrueCondition(
            condition
        );
        Parent.FindAll(TreeScope_Children, condition, elements);
        elements.Get_Length(j);
        for i := 0 to j-1 do begin
            elements.GetElement(i, child);
            child.Get_CurrentName(ws);

            if ws <> '' then begin
                if str <> '' then str := str + '  ';
                str := str + ws;
            end;
        end;
        result := str <> '';
        if result then s := str;
    end;
end;
class function TUIAutomation.GetColumnIndex(c : TTICInfo; var idx : integer) : boolean;
var
    uiaType : UIA_ControlType;
    element, header, child : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    i,j : integer;
    same : longBool;
begin
    result := false;
    a := GetAutomation;
    a.CreateTrueCondition(condition);
    a.Get_ControlViewWalker(tw);

    element := c.UIA.Element;
    uiaType :=  GetProperty(element, UIA_ControlTypePropertyId);
    case uiaType of
    UIA_HeaderControlTypeId:
        EXIT;
    UIA_HeaderItemControlTypeId:
        begin
            tw.GetParentElement(element, header);
        end;
    else
        begin
            tw.GetParentElement(element, header);
        end;
    end;

    uiaType :=  GetProperty(header, UIA_ControlTypePropertyId);
    if (uiaType = UIA_HeaderControlTypeId) then begin
        header.FindAll(TreeScope_Children, condition, elements);
        elements.Get_Length(j);
        for i := 0 to j-1 do begin
            elements.GetElement(i, child);
            a.CompareElements(child, element, integer(same));
            if same then begin
                result := true;
                idx := i;
                EXIT;
            end;
        end;
    end;
end;


class function TUIAutomation.GetAssociatedListView(header : TTICInfo; var listview : TTICInfo) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, parent, tempparent : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    i,j : integer;
    same : longBool;
    GridPattern : IUIAutomationGridPattern;
    ValPattern : IUIAutomationValuePattern;
    ws : WideString;
begin
    result := false;
    a := GetAutomation;
    a.CreateTrueCondition(condition);
    a.Get_ControlViewWalker(tw);

    element := header.UIA.Element;

    tw.GetParentElement(element, parent);
    uiaType :=  GetProperty(parent, UIA_ControlTypePropertyId);
    if (uiaType <> UIA_ListControlTypeId) then begin
        tw.GetParentElement(parent, tempparent);
        parent := tempparent;
        uiaType :=  GetProperty(tempparent, UIA_ControlTypePropertyId);
    end;
    result := (uiaType = UIA_ListControlTypeId);
    if result then begin
        listview := TTICInfo.Create;
        FillInfo(listview, parent);
    end;
end;

class function TUIAutomation.GetListViewTextAsGrid(
c : TTICInfo; row, col : integer; var s : string) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, child : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    i,j : integer;
    same : longBool;
    str : string;

    function GetGridText(
        element : IUIAutomationElement; row,col : integer; var s : string) : boolean;
    var
        GridPattern : IUIAutomationGridPattern;
        ValPattern : IUIAutomationValuePattern;
        ws : WideString;
        tempEelment : IUIAutomationElement;
    begin
        result := false;
        result := GetProperty(element, UIA_IsGridPatternAvailablePropertyId);
        if not result then EXIT;
        if not TUIAutomation.TryAndGet(
            element,
            UIA_GridPatternId,
            IUnknown(GridPattern)
        ) then EXIT;

        GridPattern.GetItem(row, col, tempElement);
        if TUIAutomation.TryAndGet(
            tempElement,
            UIA_ValuePatternId,
            IUnknown(ValPattern)
        ) then begin
            ValPattern.Get_CurrentValue(ws);
            result := true;
            if ws <> '' then begin
               s := ws;
            end;
        end;
    end;
begin
    result := false;
    a := GetAutomation;
    a.CreateTrueCondition(condition);
    a.Get_ControlViewWalker(tw);

    element := c.UIA.Element;
    if GetGridText(element, row, col, str) then begin
        s := str;
        result := true;
    end else begin
        // TODO: find a test case where this will work
        //
        result := TUIAutomation.GetListViewRow(
            c, row, element
        );
        if not result then EXIT;
        result := TUIAutomation.FindChildByIndex(
            element, UIA_UNKNOWN, col, child
        );

        if not result then EXIT;
        result := GetTextOrLink(tempelement, s);
    end;

end;
class function TUIAutomation.GetListViewRowCount(c : TTICInfo; var count : integer) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, parent, tempparent : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    GridPattern : IUIAutomationGridPattern;
begin
    result := false;
    element := c.UIA.Element;
    result := GetProperty(element, UIA_IsGridPatternAvailablePropertyId);
    if not result then EXIT;
    if not TUIAutomation.TryAndGet(
        element,
        UIA_GridPatternId,
        IUnknown(GridPattern)
    ) then EXIT;

    result := GridPattern.Get_CurrentRowCount(count)= S_OK;
end;
class function TUIAutomation.GetListViewRow(c : TTICinfo; var row : integer) : boolean;
var
    element : IUIAutomationElement;
begin
    TUIAutomation.GetListViewRow(c, row, element);
end;
class function TUIAutomation.GetListViewRow(c : TTICinfo; var row : integer; var listItem : IUIAutomationElement) : boolean;
var
    uiaType : UIA_ControlType;
    element, listElement, tempelement, parent, child : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    i,j,k, rows, cols : integer;
    same : longBool;
    GridPattern : IUIAutomationGridPattern;
    GridItemPattern : IUIAutomationGridItemPattern;
    ValPattern : IUIAutomationValuePattern;
    str : string;

    function ListItemToControlParent(element : IUIAutomationElement; var uiaType : UIA_COntrolType) : IUIAutomationElement;
    begin
        result := nil;
        uiaType := GetProperty(element, UIA_ControlTypePropertyId);
        if (uiaType <> UIA_ListControlTypeId) then begin
            repeat
            tw.GetParentElement(element, parent);
            element := parent;
            uiaType := GetProperty(element, UIA_ControlTypePropertyId);
            until (uiaType = UIA_ListControlTypeId);

            result := element;
        end else begin
            result := element;
        end;
    end;
begin
    result := false;
    a := GetAutomation;
    a.Get_ControlViewWalker(tw);

    element := c.UIA.Element;
    listElement := ListItemToControlParent(element, uiaType);

    j := -1;
    if (uiaType = UIA_ListControlTypeId) then begin
        // handle case where item is child of listitem
        if c.UIA.ParentCType = UIA_ListItemControlTypeId then begin
            tw.GetParentElement(element, parent);
            element := parent;
        end;

        // look for griditem patterns to identify row
        if TryAndGet(element, UIA_GridItemPatternId, IUnknown(GridItemPattern)) then begin
            GridItemPattern.Get_CurrentRow(j);
            listItem := element;
        // look for grid pattern of
        end else if TryAndGet(listElement, UIA_GridPatternId, IUnknown(GridPattern)) then begin
            GridPattern.Get_CurrentRowCount(rows);
            GridPattern.Get_CurrentColumnCount(cols);
            for i := 0 to rows-1 do begin
                for k := 0 to cols-1 do begin
                    GridPattern.GetItem(i,k, tempelement);
                    a.CompareElements(element, tempelement, Integer(same));
                    if same then begin
                        j := k;
                        listItem := tempelement;
                        BREAK;
                    end;
                end;
            end;
        end else begin
            // manually walk each row and use bounding rects to ID
            if FindChildByPoint(
                listElement, UIA_ListItemControlTypeId,
                c.MousePoint, child, j
            ) then begin
                listItem := child;
            end;

        end;
    end;

    result := j <> -1;
    if result then begin
        row := j;
    end;
end;
class function TUIAutomation.GetListViewSelectedCount(c : TTICINfo; var count : integer) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, parent, tempparent : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    SelectionItemPattern : IUIAutomationSelectionItemPattern;
    SelectionPattern : IUIAutomationSelectionPattern;
    hasPattern : boolean;
    tempInfo : TTICInfo;
begin
    result := false;
    element := c.UIA.Element;

    result := GetProperty(element, UIA_IsSelectionPatternAvailablePropertyId);
    if result then begin
        result := TryAndGet(element, UIA_SelectionPatternId, IUnknown(SelectionPattern));
        if not result then EXIT;

        SelectionPattern.GetCurrentSelection(elements);
        elements.Get_Length(count);
    end else begin
        case c.UIA.ParentCType  of
        UIA_ListControlTypeId,
        UIA_ListItemControlTypeId:
            begin
                tempInfo := self.GetParent(c, false);
                result := GetListViewSelectedCount(tempInfo, count);
                myfree(tempInfo);
            end;
        end;
    end;
end;


class function TUIAutomation.GetListViewSelectedRows(c : TTICINfo; rows : TList< TAutomationList>) : boolean;
begin
    result := TUIAutomation.GetListViewSelectedRows(c, -1, rows);
end;
class function TUIAutomation.GetListViewSelectedRows(c : TTICINfo; col : integer; rows : TList<TAutomationList>) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, child : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    SelectionPattern : IUIAutomationSelectionPattern;
    tempInfo : TTICInfo;
    i,j, cnt : integer;
    containers, b : boolean;
    str : string;
    sl : TList<string>;
begin
    result := false;
    element := c.UIA.Element;
    if element = nil then EXIT;

    result := GetProperty(element, UIA_IsSelectionPatternAvailablePropertyId);
    if result then begin
        result := TryAndGet(element, UIA_SelectionPatternId, IUnknown(SelectionPattern));
        if not result then EXIT;

        SelectionPattern.GetCurrentSelection(elements);
        elements.Get_Length(cnt);

        for i := 0 to cnt-1 do begin
            sl := TList<String>.Create;
            elements.GetElement(i, tempelement);
            tempInfo := TTICInfo.Create;
            FillInfo(tempInfo, tempelement);

            containers := TUIAutomation.GetChildrenTextFromContainer(
                tempInfo, sl
            );
            if (i=0) and not containers then begin
                rows.Clear;
                myfree(sl);
                result := false;
                EXIT;
            end;

            if containers and (col <> -1) then begin
               str := sl[col];
               sl.Clear;
               sl.Add(str);
            end;
            rows.Add(sl);
            myfree(tempInfo);
        end;
    end else begin
        case c.UIA.ParentCType  of
        UIA_ListControlTypeId,
        UIA_ListItemControlTypeId:
            begin
                tempInfo := self.GetParent(c, false);
                result := GetListViewSelectedRows(tempInfo, rows);
                myfree(tempInfo);
            end;
        end;
    end;
end;
class function TUIAutomation.GetTreeViewPathText(treenode : TTICInfo; var s : string) : boolean;
var
    uiaType : UIA_ControlType;
    element, tempelement, parent, tempparent : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    a : IUIAutomation;
    tw : IUIAutomationTreeWalker;
    condition : IUIAutomationCondition;
    temps, str : string;
    r : TRect;
    val : integer;
    cnt : integer;
begin
    //
    result := false;
    element := treenode.UIA.Element;
    if element = nil then EXIT;

    a := GetAutomation;
    a.CreateTrueCondition(condition);
    a.Get_ContentViewWalker(tw);

    uiaType := GetProperty(element, UIA_ControlTypePropertyId);
    temps := GetProperty(element, UIA_NamePropertyId);
    cnt := 0;
    while uiaType = UIA_TreeItemControlTypeId do begin
        inc(cnt);
        result := true;

        tw.GetParentElement(element, parent);
        element := parent;

        uiaTYpe := UIA_UNKNOWN;
        if element <> nil then begin

            uiaType := GetProperty(element, UIA_ControlTypePropertyId);
            if (uiaType = UIA_TreeItemControlTypeId) then begin
                str := GetProperty(element, UIA_NamePropertyId);
                if str <> '' then begin
                    temps := str + '  ' + temps;
                end;
            end;
        end
    end;

    if cnt = 1 then begin
        result := false;
    end;
    if result then s := temps
end;
class function TUIAutomation.GetComboBoxList(combo : TTICInfo; var s : string) : boolean;
var
    a : IUIAUtomation;
    condition : IUIAutomationCondition;
    element, parent, child, tempChild : IUIAutomationElement;
    elements : IUIAutomationElementArray;
    tw : IUIAutomationTreeWalker;
    i, j : integer;
    str : string;
    uiaType : UIA_ControlType;
begin
    result := false;
    a := GetAutomation;
    a.CreatePropertyCondition(
        Integer(UIA_ControlTypePropertyId),
        UIA_ListItemControlTypeId,
        condition
    );
    a.CreateTrueCondition(condition);
    a.Get_ControlViewWalker(tw);


    element := combo.UIA.Element;
    uiaType := GetProperty(element, UIA_ControlTypePropertyId);
    if (uiaType = UIA_EditControlTypeId) then begin
        tw.GetParentElement(element, parent);
       element := parent;
       uiaType := GetProperty(element, UIA_ControlTypePropertyId);
    end;


    element.FindAll(TreeScope_Descendants, condition, elements);
    elements.Get_Length(j);

    for i := 0 to j-1 do begin
        result := true;
        elements.GetElement(i, child);
        uiaType := GetProperty(child, UIA_ControlTypePropertyId);
        str := GetProperty(child, UIA_NamePropertyId);
        AppendX(s, str, #13#10);
    end;
end;

class function TUIAutomation.FindControl(c : TTICInfo) : boolean;
var Automation : IUIAutomation;
    tempElement : IUIAutomationElement;
    bool : longBool;
    s : string;
begin
    result := false;
    Automation := GetAutomation;
    if not Assigned(automation) then EXIT;

    try
        OleCheck(Automation.ElementFromPoint(tagPoint(c.MousePoint), tempElement));
        if assigned(tempElement) then begin
            c.UIA.ElementUnderMouse := tempElement;
            result := true;
        end;
    except on e : exception do
        begin
            s := e.Message;
        end;
    end;
end;
class procedure TUIAutomation.InspectControl(c : TTICInfo);
var Automation : IUIAutomation;
    cache : IUIAutomationCacheRequest;
    condition : IUIAutomationCondition;
    uiaType : UIA_ControlType;


    function TraversePane(
        element : IUIAutomationElement; pt : tagPoint;
        var found : IUIAutomationElement; var rect : TRect
    ) : boolean;
    var
        uiaType : UIA_ControlType;
        i, j : integer;
        r : tagRect;
        elements : IUIAutomationElementArray;
        child : IUIAutomationElement;
        checkPoint : boolean;
    begin
        // check for non-containers on the way down
        // check for containters on the way up if child not found

        result := false;

        element.FindAllBuildCache(
            TreeScope_Children, condition, cache, elements
        );
        elements.Get_Length(j);
        for i := 0 to j-1 do begin
            elements.GetElement(i, child);
            uiaType := GetProperty(cache, child, UIA_ControlTypePropertyId);

            checkPoint := not isContainer(uiaType);
            if not checkPoint then begin
                result := TraversePane(child, pt, found, rect);
                checkPoint := not result;
            end;
            if checkPoint then begin
                child.Get_CachedBoundingRectangle(r);
                if PtInRect(trect(r), tpoint(pt)) then begin
                    found := child;
                    rect := trect(r);
                    result := true;
                end;
            end;
            if result then BREAK;
        end;
    end;
begin
    Automation := GetAutomation;
    if not Assigned(automation) then EXIT;
    try
        uiaType := GetProperty(c.UIA.ElementUnderMouse, UIA_ControlTypePropertyId);
        if isContainer(uiaType) then begin
            Automation.CreatePropertyCondition(
                Integer(UIA_IsContentElementPropertyId),
                true,
                condition
            );

            automation.CreateCacheRequest(cache);
            cache.AddProperty(Integer(UIA_ControlTypePropertyId));
            cache.AddProperty(Integer(UIA_BoundingRectanglePropertyId));
            cache.Set_TreeScope(TreeScope_Element);

            if TraversePane(
                c.UIA.ElementUnderMouse, tagPoint(c.MousePoint),
                c.UIA.Element, c.WindowRect
            ) then begin
                FillInfo(c, c.UIA.Element);
            end else begin
                FillInfo(c, c.UIA.ElementUnderMouse);
                c.WIndowRect := c.UIA.Rect;
            end;
        end else begin
            FillInfo(c, c.UIA.ElementUnderMouse);
            c.WindowRect := c.UIA.Rect;
        end;
    except

    end;
end;
class procedure TUIAutomation.FillInfo(C : TTICInfo; element : IUIAutomationElement);
var i,j,k : integer;
    r : tagRect;
    ws, ws2 : WideString;
    ValPattern  : IUIAutomationValuePattern;
    pTextRange : pointer;
    elements : IUIAutomationElementArray;
    parent, tempElement, gridElement : IUIAutomationElement;
    tempC : TTICInfo;
    a : IUIAutomation;
    cache : IUIAutomationCacheRequest;
    condition : IUIAutomationCondition;
    s : string;
    pt : TPoint;
    ovar : OleVariant;
    tw : IUIAutomationTreeWalker;
    uiaType : UIA_ControlType;
begin
    try
        a := GetAutomation;
        a.CreateCacheRequest(cache);
        cache.AddProperty(Integer(UIA_NativeWindowHandlePropertyId));
        cache.AddProperty(Integer(UIA_ControlTypePropertyId));
        cache.AddProperty(Integer(UIA_AutomationIdPropertyId));
        cache.AddProperty(Integer(UIA_NamePropertyId));
        cache.AddProperty(Integer(UIA_ClassNamePropertyId));
        cache.AddProperty(Integer(UIA_ControlTypePropertyId));
        cache.AddProperty(Integer(UIA_BoundingRectanglePropertyId));
        cache.AddProperty(Integer(UIA_ValueValuePropertyId));
        cache.AddProperty(Integer(UIA_ClickablePointPropertyId));
        cache.AddProperty(Integer(UIA_IsKeyboardFocusablePropertyId));
        cache.AddPattern(Integer(UIA_ValuePatternId));
        cache.Set_TreeScope(TreeScope_Element);

        a.CreateTrueCondition(condition);
        element.FindFirstBuildCache(TreeScope_Element, condition, cache, tempElement);
        element := tempElement;

        c.UIA.Element := element;
        c.UIA.Handle := GetProperty(cache, element, UIA_NativeWindowHandlePropertyId);
        fillchar(r, sizeof(r), #0);
        element.Get_CachedBoundingRectangle(r);
        c.UIA.CType := GetProperty(cache,element, UIA_ControlTypePropertyId);
        c.UIA.Id := GetProperty(cache,element, UIA_AutomationIdPropertyId);
        c.UIA.IsFocusable := GetProperty(cache,element, UIA_IsKeyboardFocusablePropertyId);

        element.Get_CachedName(ws); // backup name
        if (ws <> '') then begin
            c.UIA.Name := ws;
        end else begin
            ws := GetProperty(cache, element, UIA_NamePropertyId);
            c.UIA.Name := ws;
        end;

        element.Get_CachedClassName(ws2);
        c.UIA.Classname := ws2;
        if c.ClassName = '' then begin
            c.ClassName :=c.UIA.Classname;
        end;

        if TUIAutomation.TryAndGet(
            element,
            UIA_ValuePatternId,
            IUnknown(ValPattern)
        ) then begin
            ValPattern.Get_CachedValue(ws2);
            if (ws = '') and (ws2 <> '') then begin
                ws := ws2;
            end;
        end;

        a.Get_ControlViewWalker(tw);
        tw.GetParentElement(element, parent);
        c.UIA.ParentCType := GetProperty(parent, UIA_ControlTypePropertyId);

        case c.UIA.CType of
        UIA_EditControlTypeId:
            begin
                GetTextOrLink(element, s);
                if s <> '' then ws := s;
            end;
        UIA_HyperlinkControlTypeId:
            begin
                ws2 := GetProperty(cache, element, UIA_ValueValuePropertyId);
                ws := ws + #13#10 + ws2;
            end;
        UIA_CustomControlTypeId:
            begin
                if GetCustomText(element, c.MousePoint, c.WindowRect, s) then begin
                    ws := s;
                    r := tagRect(c.WindowRect);
                end;
            end;
        UIA_ListItemControlTypeId:
            begin
                if GetListItemText(element, c.MousePoint, s) then begin
                    ws := s;
                end;
            end;
        UIA_TextControlTypeId:
            begin
                if GetTextControlText(element, s) then begin
                    ws := s;
                end;
            end
        else
            begin
                if GetGridRowText(element, s) then begin
                    ws := s;
                end else if GetHeaderText(element, s) then begin
                    ws := s;
                end;
            end;
        end;


        c.UIA.Rect := rect(r.left,r.top,r.right,r.bottom);

        if (ws <> '') then begin
            c.windowtext := ws;
        end;
    except

    end;
end;

class function TUIAutomation.GetChildrenTextFromContainer(c : TTICInfo; textList : TList<string>) : boolean;
var
    a : IUIAutomation;
    condition : IUIAutomationCondition;
    elements : IUIAutomationElementArray;
    child : IUIAutomationElement;
    i, j : integer;
    str : string;
    uiaType : UIA_ControlType;
begin
    a := GetAutomation;
    a.CreateTrueCondition(condition);

    c.uia.element.FindAll(TreeScope_Children, condition, elements);
    elements.Get_Length(j);
    for i := 0 to j-1 do begin
        result := true;
        elements.GetElement(i, child);
        str := '';
        uiaType := GetProperty(child, UIA_ControlTypePropertyId);

        case uiaType of
        UIA_EditControlTypeId,
        UIA_HyperlinkControlTypeId,
        UIA_TextControlTypeId,
        UIA_ListItemControlTypeId,
        UIA_CheckBoxControlTypeId,
        UIA_RadioButtonControlTypeId,
        UIA_ComboBoxControlTypeId:
            begin
                GetTextOrLink(child, str);
                textList.Add(str);
            end;
        end;
    end;

end;
class function TUIAutomation.GetIdProperty(h : THandle) : string;
var a : IUIAutomation;
    e : IUIAutomationElement;
begin
    result := '';
    a := GetAutomation;
    if a = nil then EXIT;

    a.ElementFromHandle(Pointer(h), e);
    if e = nil then EXIT;

    result := GetProperty(e, UIA_AutomationIdPropertyId);
end;
class function TUIAutomation.HandleFromPoint(pt : TPoint) : THandle;
var a : IUIAutomation;
    p : tagPoint;
    e : IUIAutomationElement;
begin
    result := 0;
    a := GetAutomation;
    if not assigned(a) then EXIT;

    p.x := pt.X;
    p.y := pt.Y;
    a.ElementFromPoint(p, e);
    if not assigned(e) then EXIT;

    result := getProperty(e, UIA_NativeWindowHandlePropertyId);
end;
class function TUIAutomation.GetFullPathName(h : THandle) : string;
var a : IUIAutomation;
    e : IUIAutomationElement;
begin
    result := '';
    a := GetAutomation;
    if not assigned(a) then EXIT;


    a.ElementFromHandle(Pointer(h), e);
    if not assigned(e) then EXIT;

    result := self.GetFullPathName(e);
end;
class function TUIAutomation.GetParent(c : TTICInfo; Content : boolean = true) : TTICinfo;
var a : IUIAutomation;
    e : IUIAutomationElement;
    tw : IUIAutomationTreeWalker;
begin
    result := nil;
    a := GetAutomation;
    if not assigned(a) then EXIT;

    if content then begin
        a.Get_ContentViewWalker(tw);
    end else begin
        a.Get_ControlViewWalker(tw);
    end;
    if not assigned(tw) then EXIT;

    tw.GetParentElement(c.UIA.Element, e);
    if not assigned(e) then EXIT;

    result := TTICInfo.Create;
    FillInfo(result, e);
end;
class function TUIAutomation.FocusControl(path : string) : boolean;
var
    element : IUIAutomationElement;
begin
    result := TUIAutomation.FullPathNameToElement(path, element);
    if not result then EXIT;

    result := Element.SetFocus = S_OK;
end;

initialization
finalization
begin
    TFrmDebug.MeOnlyAppend('UnitUIAutomation', false);
end;

end.
