unit UnitViewExtract;
{

Working with external view sucks.

Issues:
- messages expect all pointers to be in the target process' memory space
- ListViews in 64-bit processes use a different "LVITEM" struct because of pointer sizes
- processes that are DPI aware need to be scaled to coordinates
that AC understands.



Workarounds:
- data is injected into the target process for ListView messages
- points for LVM_HITTEST and LVM_GETITEMRECT have to be scaled from
the virtual desktop and the native desktop resolution
- a special case needs to be made for retreiving text from a 64-bit listview
}

interface

uses System.Types, Generics.collections;


type TExtractList = TList<String>;
type THeaderExtract = class(TObject)
    private

    class function GetColumnAt(h : THandle; p : TPoint) : integer; overload;
    class function GetColumnAt(h : THandle; p : TPoint; var rect : TRect) : integer; overload;
    class function GetItemText(h : THandle; idx : integer) : string;
    class function GetItemText64(h : THandle; idx : integer) : string;
    class function GetRect(h : Thandle; idx : integer; var subrect : TRect) : boolean;

    public
    class function GetColumnIdx(h : THandle;  pt : TPoint) : integer;
    class function GetText(h : THandle; idx : integer) : string; overload;
    class function GetText(h : THandle; pt : TPoint; var subrect: TRect) : string; overload;
    class function GetColumnCount(h : THandle) : integer;
end;
type TListViewExtract = class(TObject)
    private
    class function GetListViewText(h : THandle; row, column : integer) : string; overload;
    class function GetListViewText(h : THandle; row : integer) : string; overload;
    class function GetListViewText64(h : THandle; row : integer) : string; overload;
    class function GetListViewText64(h : THandle; row, column : integer) : string; overload;
    class function GetListViewRect(h : THandle; row : integer ) : TRect;
    class function GetListViewPoint(h : Thandle; row : integer) : TPoint;
    class function GetListViewItemAt(h : THandle; p : TPoint ) : integer;

    public
    class function GetText(h : THandle; pt : Tpoint; var subrect : Trect; textList : TList<string>) : boolean; overload;
    class function GetText(h : THandle; row, col : integer; var s : string) : boolean; overload;
    class function GetText(h : THandle; row : integer; textList : TList<string>) : boolean; overload;

    class function GetListViewItemAtSlow(h : THandle; pt : TPoint; var row : integer) : boolean;

    class function GetHeader(h : THandle) : THandle;
    class function GetListViewColumnCount(h : THandle) : integer;
    class function GetListViewRowCount(h : THandle) : integer;

    class function GetListViewView(h : THandle) : integer;

    class function GetSelectedCount(h : THandle) : integer;
    class function GetSelectedRows(h: THandle; rows : TList<TExtractList>) : boolean; overload;
    class function GetSelectedRows(h: THandle; col : integer; rows : TList<TExtractList>) : boolean; overload;
end;
type TTreeViewExtract = class(TObject)
    private
    public
    class function GetPathText(h : THandle; pt : TPoint; var s : string) : boolean;
    class function GetText(h : THandle; pt : TPoint; var s : string) : boolean;
end;
type TListBoxExtract = class(TObject)
    public
    class function GetCount(h : THandle; var cnt : integer) : boolean;
    class function GetRow(h : THandle; pt : TPoint; var row : integer) : boolean;
    class function GetText(h : THandle; row : integer; var s : string) : boolean; overload;
    class function GetText(h : THandle; row : integer; var s : string; var subrect : TRect) : boolean; overload;
    class function GetSelected(h : THandle; var rows : TExtractList) : boolean;
    class function GetSelectedCount(h : THandle; var rows : integer) : boolean;
end;
implementation

uses Winapi.Windows, Winapi.messages, Winapi.CommCtrl,
Forms, UnitMisc, Unit64BitDetect, UnitMouseScaler,
SysUtils,  Winapi.DwmApi, UnitFrmDebug;


//
// nice little macro to send messages with remote
// address spaced structs
//
type TMessage = class(TObject)
    public
    class function Send<T>(
        procH, h : THandle;
        msg : integer; lparam : integer; var wparam : T) : integer; inline;
end;
class function TMessage.Send<T>(
    procH, h : THandle;
    msg : integer; lparam : integer; var wparam : T) : integer;
var lvItemP : Pointer;
    sz : NativeUInt;

begin
    lvItemP := WinApi.Windows.VirtualAllocEx(
        procH,
        nil,
        sizeof(wparam),
        MEM_COMMIT,
        PAGE_READWRITE
    );
    WriteProcessMemory(
        Long(procH),
        lvItemP,
        @wparam,
        sizeof(wparam),
        sz
    );
    result := SendMessage(h, MSG, lparam, Integer(lvItemP));
    ReadProcessMemory(
        procH,
        lvItemP,
        @wparam,
        sizeof(wparam),
        sz
    );
    VirtualFreeEx(procH, lvItemP, 0, MEM_RELEASE);
end;


//
// Best effort to detect if a process requires coordinates
// scaled to native resolution.
//
// AC is not DPI aware and will run virtualized on a system
// that has scaling. Sending messages to a DPI aware process
// requires converting to native for Send and from native for Receive
//

type PROCESS_DPI_AWARENESS = (
    PROCESS_DPI_UNAWARE = 0,
    PROCESS_SYSTEM_DPI_AWARE = 1,
    PROCESS_PER_MONITOR_DPI_AWARE = 2
);

var
GetProcessDpiAwareness: function(
    hprocess: THANDLE;
    out value: PROCESS_DPI_AWARENESS
): HRESULT; stdcall;

procedure RtlGetNtVersionNumbers(
    out MajorVersion : DWORD;
    out MinorVersion : DWORD;
    out Build        : DWORD
); stdcall; external 'ntdll.dll';
type TScale = class(Tobject)
    public
        class function NeedsScaling(h : THandle) : boolean;
end;
class function TScale.NeedsScaling(h : THandle) : boolean;
var r, r2 : TRect;
    scale : double;
    dc : THandle;
    libh : THandle;
    i, wd, wd2 : integer;
    s : string;
    awareness : PROCESS_DPI_AWARENESS;
    maj, min, build : cardinal;
    procID : cardinal;
    procH : Thandle;
begin
    result := false;

    // detect is scaling is enabled
    // AC is not aware, so "Desktop" will use virtual coordinates
    dc := GetDC(0);
    wd := GetDeviceCaps(dc, DESKTOPHORZRES);
    wd2 := screen.DesktopWidth;
    if wd = wd2 then EXIT;

    // can't use Win32MajVersion, since Windows uses virtualization
    // based on target OS (version 6 for Delphi XE2)

    RtlGetNtVersionNumbers(maj, min, build);
    if (maj = 8) and (min < 1) then EXIT;
    if (maj >= 8) then begin
        GetWindowThreadProcessId(h, procID);
        procH := OpenProcess(
            PROCESS_QUERY_INFORMATION,
            false,
            procID
        );


        libH := LoadLibrary('shcore.dll');
        @GetProcessDpiAwareness := GetProcAddress(libH, 'GetProcessDpiAwareness');

        GetProcessDpiAwareness(procH, awareness);

        FreeLibrary(libH);

        CloseHandle(procH);
        result := awareness <> PROCESS_DPI_UNAWARE;
    end;
end;



class function TListViewExtract.GetText(h : THandle;  pt : Tpoint; var subrect : TRect; textList : TList<string>) : boolean;
var i, idx, iView : integer;
    s, coltext : string;
    scaling : boolean;
    is64 : boolean;
    r : trect;
    headerH : THandle;
    col : integer;
    mouse : TPoint;
begin
    result := false;
    scaling := TScale.NeedsScaling(h);
    is64 := T64BitDetect.Is64Bit(h);

    mouse := pt;
    Winapi.Windows.ScreenToClient(h, mouse);
    if scaling then begin
        mouse := TMouseScaler.ScalePoint(mouse, true);
    end;

    idx := TListViewExtract.GetListViewItemAt(h, mouse);
    if (idx = -1) then begin
        for i := 0 to TListViewExtract.GetListViewRowCount(h)-1 do begin
            r := TListViewExtract.GetListViewRect(h, i);
            if (mouse.Y >= r.Top) and (mouse.Y <= r.Bottom) then begin
                idx := i;
                BREAK;
            end;
        end;
    end;
    if idx = -1 then EXIT;

    pt := TListViewExtract.GetListViewPoint(h, idx);
    if scaling then begin
        pt := TMouseScaler.ScalePoint(pt, false);
    end;
    subRect := TListViewExtract.GetListViewRect(h, idx);
    if scaling then begin
        subRect.TopLeft := TMouseScaler.ScalePoint(SubRect.TopLeft, false);
        subRect.BottomRight := TMouseScaler.ScalePoint(SubRect.BottomRight, false);
    end;
    MapWindowPoints(h, 0, subrect, 2);

    headerH := TListViewExtract.GetHeader(h);
    col := -1;
    if (headerH <> 0) then begin
        for i := 0 to THeaderExtract.GetColumnCount(headerH)-1 do begin
            THeaderExtract.GetRect(headerH, i, r);
            if scaling then begin
                r.TopLeft := TMouseScaler.ScalePoint(r.TopLeft, false);
                r.BottomRight := TMouseScaler.ScalePoint(r.BottomRight, false);
            end;
            //MapWindowPoints(headerH, 0, r, 2);

            if (mouse.X >= r.left) and (mouse.X <= (r.Left+r.Width)) then begin
                col := i;
                BREAK;
            end;
        end;
    end;

    result := true;
    if (is64) then begin
        iView := TListViewExtract.GetListViewView(h);
        if (iView = LV_VIEW_DETAILS) then begin
            if col = -1 then begin
                for i := 0 to TListViewExtract.GetListViewColumnCount(h) - 1 do begin
                    textList.Add(
                        TListViewExtract.GetListViewText64(h, idx, i)
                    );
                end;
            end else begin
                textList.Add(
                    TListViewExtract.GetListViewText64(h, idx, col)
                );

            end;
        end else begin
            textList.Add(
                TListViewExtract.GetListViewText64(h, idx)
            );
        end;
    end else begin
        iView := TListViewExtract.GetListViewView(h);
        if (iView = LV_VIEW_DETAILS) then begin
            if col= -1 then begin
                for i := 0 to TListViewExtract.GetListViewColumnCount(h) - 1 do begin
                    textList.Add(TListViewExtract.GetListViewText(h, idx, i));
                end;
            end else begin
                textList.Add(TListViewExtract.GetListViewText(h, idx, col));
            end;
        end else begin
            textList.add(TListViewExtract.GetListViewText(h, idx) );
        end;
    end;
end;
class function TListViewExtract.GetText(h : THandle; row : integer; textList : TList<string>) : boolean;
var i, idx, iView : integer;
    s, coltext : string;
    is64 : boolean;
begin
    result := false;
    is64 := T64BitDetect.Is64Bit(h);

    idx := row;
    result := true;
    if (is64) then begin
        iView := TListViewExtract.GetListViewView(h);
        if (iView = LV_VIEW_DETAILS) then begin
            for i := 0 to TListViewExtract.GetListViewColumnCount(h) - 1 do begin
                textList.Add(
                    TListViewExtract.GetListViewText64(h, idx, i)
                );
            end;
        end else begin
            textList.Add(
                TListViewExtract.GetListViewText64(h, idx)
            );
        end;
    end else begin
        iView := TListViewExtract.GetListViewView(h);
        if (iView = LV_VIEW_DETAILS) then begin
            for i := 0 to TListViewExtract.GetListViewColumnCount(h) - 1 do begin
                textList.Add(TListViewExtract.GetListViewText(h, idx, i));
            end;
        end else begin
            textList.add(TListViewExtract.GetListViewText(h, idx) );
        end;
    end;
end;
class function TListViewExtract.GetText(h : THandle; row, col : integer; var s : string) : boolean;
var
    is64 : boolean;
begin
    is64 := T64BitDetect.Is64Bit(h);
    if (is64) then begin
        s := TListViewExtract.GetListViewText64(h, row, col);
    end else begin
        s := TListViewExtract.GetListViewText(h, row, col);
    end;
    result := true;
end;

// NOTE: This record is likely to have errors and was derived from
// an unreliable source. However, it does work for the two LMV_ messages
// used here.

type LV_ITEM64 = record
    mask : Cardinal;
    iItem : Integer;
    iSubItem : Integer;
    state : Cardinal;
    stateMask : Cardinal;
    padding1 : integer;

    pszText : PWideChar;
    padding2 : integer;

    cchTextMax : Integer;
    iImage : Integer;
    lParam : Cardinal;
    padding3 : integer;

    iIndent : integer;
    iGroupID : integer;
    cColumns : Cardinal;
    padding4 : integer;

    puColumns : Cardinal;
    padding5 : integer;

    piColFmt : Cardinal;
    padding6 : integer;

    iGroup : integer;
    padding7 : integer;
end;
const MAX_LVMSTRING_CHARS = 1024;

class function TListViewExtract.GetListViewText64(h : THandle; row : integer) : string;
begin
    GetListViewText64(h,row, -1);
end;
class function TListViewExtract.GetListViewText64(h : THandle; row, column : integer) : string;
var
    procH : THandle;
    textP : pointer;
    lv2 : LV_ITEM64;
    lvItemP : Pointer;
    sz : NativeUInt;
    localText : string;
    procID : cardinal;
    i : integer;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    textP := WinAPI.Windows.VirtualAllocEx(
        procH,
        nil,
        MAX_LVMSTRING_CHARS * sizeof(char),
        MEM_COMMIT,
        PAGE_READWRITE
    );

    FillChar(lv2, sizeof(lv2), #0);
    lv2.mask := LVIF_TEXT;
    lv2.cchTextMax := MAX_LVMSTRING_CHARS;  // characters, not bytes
    lv2.pszText :=  textP;
    lv2.iItem := row;
    if column = -1 then  begin
        lv2.iSubItem := 0;
    end else begin
        lv2.iSubItem := column;
    end;

    lvItemP := WinApi.Windows.VirtualAllocEx(
        procH,
        nil,
        sizeof(lv2),
        MEM_COMMIT,
        PAGE_READWRITE
    );
    WriteProcessMemory(
        procH, lvItemP,
        @lv2, sizeof(lv2),
        sz
    );

    i := SendMessage(h, LVM_GETITEMTEXT, row, WPARAM(lvItemP));

    SetLength(localText, MAX_LVMSTRING_CHARS);
    ReadProcessMemory(
        procH,
        textP,
        @localText[1],
        MAX_LVMSTRING_CHARS,
        sz
    );
    result := PChar(localText);


    VirtualFreeEx(procH, textP, 0, MEM_RELEASE);
    VirtualFreeEx(procH, lvItemP, 0, MEM_RELEASE);

    CloseHandle(procH);
end;

class function TListViewExtract.GetListViewText(h : THandle; row, column : integer) : string;
var
    procH : THandle;
    textP : pointer;
    lv : tagLVITEMW;
    lvItemP : Pointer;
    sz : NativeUInt;

    localText : string;
    procID : cardinal;
    i : integer;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    textP := WinAPI.Windows.VirtualAllocEx(
        procH,
        nil,
        MAX_LVMSTRING_CHARS * sizeof(char),
        MEM_COMMIT,
        PAGE_READWRITE
    );


    FillChar(lv,sizeof(lv), #0);
    lv.mask := LVIF_TEXT;
    lv.cchTextMax := MAX_LVMSTRING_CHARS;
    lv.pszText := textP;
    lv.iItem := row;
    if column = -1 then begin
        lv.iSubItem := 0;
    end else begin
        lv.iSubItem := column;
    end;

    lvItemP := WinApi.Windows.VirtualAllocEx(
        procH,
        nil,
        sizeof(lv),
        MEM_COMMIT,
        PAGE_READWRITE
    );
    WriteProcessMemory(
        Long(procH),
        lvItemP,
        @lv,
        sizeof(lv),
        sz
    );

    if (column = -1) then begin
        i := SendMessage(h, LVM_GETITEMTEXT, row, WPARAM(lvItemP));
    end else begin
        i := SendMessage(h, LVM_GETITEMW, 0, WPARAM(lvItemP));
    end;

    SetLength(localText, MAX_LVMSTRING_CHARS);
    ReadProcessMemory(
        procH,
        textP,
        @localText[1],
        MAX_LVMSTRING_CHARS,
        sz
    );
    result := PChar(localText);

    VirtualFreeEx(procH, textP, 0, MEM_RELEASE);
    VirtualFreeEx(procH, lvItemP, 0, MEM_RELEASE);

    CloseHandle(procH);
end;
class function TListViewExtract.GetListViewText(h : THandle; row : integer) : string;
begin
    result := self.GetListViewText(h, row, -1)
end;

class function TListViewExtract.GetListViewItemAt(h : THandle; p : TPoint) : integer;
var
    procH : THandle;
    procID : cardinal;
    hitP : Pointer;
    hit : tagLVHITTESTINFO;
    i : integer;
    sz : NativeUInt;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );
    fillchar(hit, sizeof(hit), #0);
    hit.pt.x :=  p.x;
    hit.pt.Y := p.Y ;
    hit.flags := LVHT_ONITEM or LVHT_NOWHERE;

    TMessage.Send<tagLVHITTESTINFO>(
        procH, h,
        LVM_HITTEST, 0, hit);

    result := hit.iitem;

    CloseHandle(procH);
end;
class function TListViewExtract.GetListViewItemAtSlow(h : THandle; pt : TPoint; var row : integer) : boolean;
var
    procH : THandle;
    procID : cardinal;
    hitP : Pointer;
    hit : tagLVHITTESTINFO;
    i, j : integer;
    sz : NativeUInt;
    r : Trect;
    scaling : boolean;
begin
    result := false;
    scaling := TScale.NeedsScaling(h);

    winapi.Windows.ScreenToClient(h, pt);
    if scaling then begin
        pt := TMouseScaler.ScalePoint(pt);
    end;
    j := TListViewExtract.GetListViewRowCount(h);
    for i := 0 to j - 1 do begin
        r := TListViewExtract.GetListViewRect(h, i);
        if PtInRect(r, pt) then begin
            row := i;
            result := true;
            EXIT;
        end;
    end;

end;

class function TListViewExtract.GetListViewColumnCount(h : THandle) : integer;
var header : THandle;
begin
    header := SendMessage(h, LVM_GETHEADER, 0, 0);
    result := 0;
    if (header <> 0) then begin
        result := SendMessage(header, HDM_GETITEMCOUNT, 0, 0);
    end;
    if result = 0 then result := 1;
end;
class function TListViewExtract.GetListViewRowCount(h : THandle) : integer;
begin
    result := SendMessage(h, LVM_GETITEMCOUNT, 0, 0);
end;
class function TListViewExtract.GetListViewRect(h : THandle; row : integer) : TRect;
var
    rectP : pointer;
    procH : THandle;
    procID : Cardinal;
    sz : nativeUInt;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    result := Trect.Empty;
    result.Left := LVIR_SELECTBOUNDS;
    TMessage.Send<TRect>(
        procH, h,
        LVM_GETITEMRECT, row, result
    );

    CloseHandle(procH);
end;

class function TListViewExtract.GetListViewPoint(h : Thandle; row : integer) : TPoint;
var
    pointP : pointer;
    procH : THandle;
    procID : Cardinal;
    sz : nativeUInt;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    result := point(0,0);
    TMessage.Send<TPoint>(
        procH, h,
        LVM_GETITEMPOSITION, row, result
    );

    CloseHandle(procH);
    MapWindowPoints(h, 0, result, 1);
end;
class function TListViewExtract.GetListViewView(h : THandle) : integer;
begin
    result := SendMessage(h, LVM_GETVIEW, 0, 0);
end;
class function TListViewExtract.GetHeader(h : THandle) : THandle;
begin
    result := SendMessage(h, LVM_GETHEADER, 0, 0);
end;

class function TListViewExtract.GetSelectedCount(h : THandle) : integer;
begin
    result := SendMessage(h, LVM_GETSELECTEDCOUNT, 0, 0);
end;
class function TListViewExtract.GetSelectedRows(h: THandle; rows : TList<TExtractList>) : boolean;
begin
    result := TListViewExtract.GetSelectedRows(h, -1, rows);
end;
class function TListViewExtract.GetSelectedRows(h: THandle; col : integer; rows : TList<TExtractList>) : boolean;
var
    i,j, idx, cnt, colcnt, x : integer;
    str : string;
begin
    cnt := TListViewExtract.GetSelectedCount(h);
    result := cnt > 0;
    if not result then EXIT;

    colcnt := TListViewExtract.GetListViewRowCount(h);
    idx := -1;
    for i := 0  to cnt-1 do begin
        x := SendMessage(h, LVM_GETNEXTITEM, idx, LVNI_SELECTED);
        idx := x;


        rows.Add( TList<string>.Create );

        if col=-1 then begin
            for j := 0 to colcnt-1 do begin
                str := TListViewExtract.GetListViewText(h,idx,j);
                rows[i].Add(str);
            end;
        end else begin
            str := TListViewExtract.GetListViewText(h,idx, col);
            rows[i].Add(str);
        end;
    end;
end;


{$A8}
type HD_ITEM64W = record
    Mask: Cardinal;
    cxy: Integer;
    pszText: PWideChar;   // starts at an 8 byte boundary
    padding1 : integer;

    hbm: HBITMAP;         // starts at an 8 byte boundary
    padding2 : integer;

    cchTextMax: Integer;
    fmt: Integer;
    lParam: LPARAM;        // starts at an 8 byte boundary
    padding3 : integer;

    iImage: Integer;
    iOrder: Integer;
    iType : Cardinal;

    padding4 : integer; // this puts pvFilter at an 8 byte boundery
    pvFilter : Pointer;
    padding5 : integer;

    state : Cardinal;
    padding6 : integer; // this makes the entire struct divisible by 8
end;
class function THeaderExtract.GetText(h : THandle;  pt : TPoint; var subrect: TRect) : string;
var idx : integer;
    parentH : Thandle;
    parentClassname : string;
    isProblemListView : boolean;
    is64 : boolean;
    scaling : boolean;
begin
    result := '';
    is64 := T64BitDetect.Is64Bit(h);
    scaling := TScale.NeedsScaling(h);
    if scaling then begin
        pt := TMouseScaler.ScalePoint(pt);
    end;

    idx := THeaderExtract.GetColumnIdx(h, pt);
    if idx = -1 then EXIT;

    THeaderExtract.GetRect(h, idx, subRect);
    if scaling then begin
        SubRect.TopLeft := TMouseScaler.ScalePoint(SubRect.TopLeft, false);
        subRect.BottomRight := TMouseScaler.ScalePoint(SubRect.BottomRight, false);
    end;
    MapWindowPoints(h, 0, subrect, 2);

    if is64 then begin
        result := THeaderExtract.GetItemText64(h, idx);
    end else begin
        result := THeaderExtract.GetItemText(h, idx);
    end;
end;
class function THeaderExtract.GetText(h : THandle; idx : integer) : string;
var
    parentH : Thandle;
    parentClassname : string;
    isProblemListView : boolean;
    is64 : boolean;
begin
    result := '';
    is64 := T64BitDetect.Is64Bit(h);

    if is64 then begin
        result := THeaderExtract.GetItemText64(h, idx);
    end else begin
        result := THeaderExtract.GetItemText(h, idx);
    end;
end;
class function THeaderExtract.GetColumnIdx(h : THandle; pt : TPoint) : integer;
var idx : integer;
    is64 : boolean;
    scaling : boolean;
begin
    result := -1;
    is64 := T64BitDetect.Is64Bit(h);
    scaling := TScale.NeedsScaling(h);

    winapi.Windows.ScreenToClient(h, pt);
    if scaling then begin
        pt := TMouseScaler.ScalePoint(pt, true);
    end;
    result := THeaderExtract.GetColumnAt(h, pt);
end;
class function THeaderExtract.GetColumnAt(h : THandle; p : TPoint; var rect : TRect) : integer;
var
    procH : THandle;
    procID : cardinal;
    hitP : Pointer;
    hit : HD_HITTESTINFO;
    i : integer;
    sz : NativeUInt;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    fillchar(hit, sizeof(hit), #0);
    hit.Point.x :=  p.x;
    hit.Point.Y := p.Y ;
    hit.flags := LVHT_ONITEM;

    TMessage.Send<HD_HITTESTINFO>(
        procH, h,
        HDM_HITTEST, 0, hit);

    result := hit.Item;

    if result <> -1 then begin
        fillchar(rect, sizeof(rect), #0);
        TMessage.send<TRect>(
            procH, h,
            HDM_GETITEMRECT, result, rect
        );

    end;
    CloseHandle(procH);
end;
class function THeaderExtract.GetColumnAt(h : THandle; p : TPoint) : integer;
var r : TREct;
begin
    result := THeaderExtract.GetColumnAt(h, p, r);
end;

class function THeaderExtract.GetRect(h : Thandle; idx : integer; var subrect : TRect) : boolean;
var
    procH : THandle;
    procID : cardinal;
    hitP : Pointer;
    hit : TRect;
    i : integer;
    sz : NativeUInt;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    fillchar(subrect, sizeof(subrect), #0);
    i := TMessage.send<TRect>(
        procH, h,
        HDM_GETITEMRECT, idx, subrect
    );

    result := i <> 0;
    CloseHandle(procH);
end;
class function THeaderExtract.GetItemText(h : THandle; idx : integer) : string;
var
    procH : THandle;
    textP : pointer;
    lv : HD_ITEM;
    lvItemP : Pointer;
    sz : NativeUInt;

    localText : string;
    procID : cardinal;
    i : integer;
begin
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    textP := WinAPI.Windows.VirtualAllocEx(
        procH,
        nil,
        MAX_LVMSTRING_CHARS * sizeof(char),
        MEM_COMMIT,
        PAGE_READWRITE
    );


    FillChar(lv,sizeof(lv), #0);
    lv.mask := HDI_TEXT ;
    lv.cchTextMax := MAX_LVMSTRING_CHARS;
    lv.pszText := textP;

    lvItemP := WinApi.Windows.VirtualAllocEx(
        procH,
        nil,
        sizeof(lv),
        MEM_COMMIT,
        PAGE_READWRITE
    );
    WriteProcessMemory(
        Long(procH),
        lvItemP,
        @lv,
        sizeof(lv),
        sz
    );

    i := SendMessage(h, HDM_GETITEM, idx, WPARAM(lvItemP));

    SetLength(localText, MAX_LVMSTRING_CHARS);
    ReadProcessMemory(
        procH,
        textP,
        @localText[1],
        MAX_LVMSTRING_CHARS,
        sz
    );
    result := PChar(localText);

    VirtualFreeEx(procH, textP, 0, MEM_RELEASE);
    VirtualFreeEx(procH, lvItemP, 0, MEM_RELEASE);

    CloseHandle(procH);
end;
class function THeaderExtract.GetItemText64(h : THandle; idx : integer) : string;
var
    procH : THandle;
    textP : pointer;
    lv : HD_ITEM64W;
    lvItemP : Pointer;
    sz : NativeUInt;

    localText : string;
    procID : cardinal;
    i : integer;
begin

    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );

    textP := WinAPI.Windows.VirtualAllocEx(
        procH,
        nil,
        MAX_LVMSTRING_CHARS * sizeof(char),
        MEM_COMMIT,
        PAGE_READWRITE
    );


    FillChar(lv,sizeof(lv), #0);
    lv.mask := HDI_TEXT ;
    lv.cchTextMax := MAX_LVMSTRING_CHARS;
    lv.pszText := textP;

    lvItemP := WinApi.Windows.VirtualAllocEx(
        procH,
        nil,
        sizeof(lv),
        MEM_COMMIT,
        PAGE_READWRITE
    );
    WriteProcessMemory(
        Long(procH), lvItemP, @lv, sizeof(lv), sz
    );

    i := SendMessage(h, HDM_GETITEM, idx, WPARAM(lvItemP));

    SetLength(localText, MAX_LVMSTRING_CHARS);
    ReadProcessMemory(
        procH, lvItemP, @lv, sizeof(lv), sz
    );
    ReadProcessMemory(
        procH, textP, @localText[1], MAX_LVMSTRING_CHARS, sz
    );
    result := PChar(localText);

    VirtualFreeEx(procH, textP, 0, MEM_RELEASE);
    VirtualFreeEx(procH, lvItemP, 0, MEM_RELEASE);

    CloseHandle(procH);

end;
class function THeaderExtract.GetColumnCount(h : THandle) : integer;
begin
   result := Winapi.Windows.SendMessage(h, HDM_GETITEMCOUNT, 0, 0)
end;



class function TTreeViewExtract.GetPathText(h : THandle; pt : TPoint; var s : string) : boolean;
var
    procH : THandle;
    lv : tagTVHITTESTINFO;
    procID : cardinal;
    hitem : HTREEITEM;
    str : string;

    function GetText(hti : HTREEITEM) : string;
    var
        item : tagTVITEM;
        sp : Pointer;
        str : string;
        sz : NativeUInt;
    const
        MAX_STR = 512;
    begin
        FillChar(item,sizeof(item), #0);
        sp :=  WinApi.Windows.VirtualAllocEx(
            procH,
            nil,
            MAX_STR * SizeOf(CHAR),
            MEM_COMMIT,
            PAGE_READWRITE
        );
        item.mask := TVIF_HANDLE or TVIF_TEXT;
        item.hItem := hti;
        item.cchTextMax := MAX_STR;
        item.pszText := sp;

        TMessage.Send<tagTVITEM>(procH, h, TVM_GETITEM, 0, item);

        setlength(str, MAX_STR + 1);
        ReadProcessMemory(
            procH,
            sp,
            @str[1],
            MAX_STR,
            sz);
        result := pchar(str);
        VirtualFreeEx(procH, sp, 0, MEM_RELEASE);
    end;
begin
    result := false;
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );


    FillChar(lv,sizeof(lv), #0);
    lv.pt := pt;
    ScreenToClient(h, lv.pt);

    TMessage.Send<tagTVHITTESTINFO>(procH, h, Integer(TVM_HITTEST), 0, lv);

    hitem := lv.hItem;
    while (hitem <> nil) do begin
        result := true;
        str := GetText(hItem);
        if str <> '' then begin
            if (s <> '') then s := '  ' + s;
            s := str + s;
        end;

        hItem := TreeView_GetParent(h, hitem);
    end;

    CloseHandle(procH);
end;
class function TTreeViewExtract.GetText(h : THandle; pt : TPoint; var s : string) : boolean;
var
    procH : THandle;
    lv : tagTVHITTESTINFO;
    procID : cardinal;
    hitem : HTREEITEM;
    str : string;
    i : integer;
    function GetText(hti : HTREEITEM) : string;
    var
        item : tagTVITEM;
        sp : Pointer;
        str : string;
        sz : NativeUInt;
    const
        MAX_STR = 512;
    begin
        FillChar(item,sizeof(item), #0);
        sp :=  WinApi.Windows.VirtualAllocEx(
            procH,
            nil,
            MAX_STR * SizeOf(CHAR),
            MEM_COMMIT,
            PAGE_READWRITE
        );
        item.mask := TVIF_HANDLE or TVIF_TEXT;
        item.hItem := hti;
        item.cchTextMax := MAX_STR;
        item.pszText := sp;

        TMessage.Send<tagTVITEM>(procH, h, TVM_GETITEM, 0, item);

        setlength(str, MAX_STR + 1);
        ReadProcessMemory(
            procH,
            sp,
            @str[1],
            MAX_STR,
            sz);
        result := pchar(str);
        VirtualFreeEx(procH, sp, 0, MEM_RELEASE);
    end;
begin
    result := false;
    GetWindowThreadProcessId(h, procID);
    procH := OpenProcess(
        PROCESS_VM_OPERATION or PROCESS_VM_READ or PROCESS_VM_WRITE,
        false,
        procID
    );


    FillChar(lv,sizeof(lv), #0);
    lv.pt := pt;
    ScreenToClient(h, lv.pt);

    i := TMessage.Send<tagTVHITTESTINFO>(procH, h, Integer(TVM_HITTEST), 0, lv);
    hitem := lv.hItem;
    if hitem <> nil then begin
        result := true;
        s := GetText(hItem);
    end;

    CloseHandle(procH);
end;


class function TListBoxExtract.GetCount(h : THandle; var cnt : integer) : boolean;
begin
    cnt := SendMessage(h, LB_GETCOUNT, 0, 0);
    result := cnt <> -1;
end;
class function TListBoxExtract.GetText(h : THandle; row : integer; var s : string; var subrect : TRect) : boolean;
var cnt, i : integer;
    insz, sz : NativeUInt;
    buff : array of char;
    scaling : boolean;
begin
    result := false;
    scaling := TScale.NeedsScaling(h);

    SendMessage(h, LB_GETITEMRECT, i, Integer(@subRect));
    if scaling then begin
        subRect.TopLeft := TMouseScaler.ScalePoint(SubRect.TopLeft, false);
        subRect.BottomRight := TMouseScaler.ScalePoint(SubRect.BottomRight, false);
    end;
    MapWindowPoints(h, 0, subrect, 2);

    sz := SendMessage(h, LB_GETTEXTLEN, row, 0);
    if sz <> 0 then sz := (sz+1); // null terminator space
    setLength(buff, sz);
    i := SendMessage(h, LB_GETTEXT, WPARAM(row), LPARAM(buff));
    result := i <> LB_ERR;
    if result then begin
        s := pchar(@buff[0]);
    end;
end;
class function TListBoxExtract.GetText(h : THandle; row : integer; var s : string) : boolean;
var r : TREct;
begin
    TListBoxExtract.GetText(h, row, s, r);
end;
class function TListBoxExtract.GetRow(h : THandle; pt : TPoint; var row : integer) : boolean;
var
    i : integer;
    point : long;
begin

    Winapi.Windows.ScreenToClient(h, pt);
    if TScale.NeedsScaling(h) then begin
        pt := TMouseScaler.ScalePoint(pt, true)
    end;

    point := MakeLong(pt.X, pt.y);
    i := SendMessage(h, LB_ITEMFROMPOINT, 0, point);
    i := LOWORD(i);
    row := i;
end;
class function TListBoxExtract.GetSelected(h : THandle; var rows : TExtractList) : boolean;
type intArray = array of integer;
var i, cnt : integer;
    sel : intArray;
    str : string;
    procID : cardinal;
    procH : THandle;

begin
    result := false;
    TListBoxExtract.GetSelectedCount(h, cnt);
    if not (cnt>0) then EXIT;

    setlength(sel, cnt);
    i := SendMessage(h, LB_GETSELITEMS, cnt, integer(@sel[0]));
    if i = LB_ERR then EXIT;

    for i := 0 to cnt-1 do begin
        TListBoxExtract.GetText(h,sel[i], str);
        rows.Add(str);
    end;
end;
class function TListBoxExtract.GetSelectedCount(h : THandle; var rows : integer) : boolean;
begin
    rows :=  SendMessage(h, LB_GETSELCOUNT, 0, 0);
    result := rows <> LB_ERR;
end;

initialization
finalization
begin
    TFrmDebug.MeOnlyAppend('UnitViewExtract', false);
end;

end.
